
Import-Module $PSScriptRoot\EnsureLocalGroup\EnsureLocalGroup.psm1 -Force
Import-Module $PSScriptRoot\UserShell\UserShell.psm1 -Force
Import-Module $PSScriptRoot\LogOffUser\LogOffUser.psm1 -Force

$ErrorActionPreference = 'Stop'

function Set-AllowAllTrustedApps {
  $appxRegistryPath = "HKLM:\Software\Policies\Microsoft\Windows\Appx"
  $allowAllTrustedAppsRegistryArguments = @{
    Path = $appxRegistryPath;
    Name = "AllowAllTrustedApps";
    Value = "1";
    PropertyType = "DWORD";
  }
  if (-not (Test-Path $appxRegistryPath)) {
    New-Item -Path $appxRegistryPath -Force | Out-Null
  }
  New-ItemProperty @allowAllTrustedAppsRegistryArguments -Force | Out-Null
}

function Get-UserCredentialsFromEnvironment() {
  $Username = $env:KioskUsername
  if ($env:KioskUserPassword) {
    $Password = $env:KioskUserPassword
  } else {
    $Password = [System.Environment]::GetEnvironmentVariable("KioskUserPassword", [System.EnvironmentVariableTarget]::User)
  }
  if ($Password) {
    $SecurePassword = ConvertTo-SecureString $Password -AsPlainText -Force
  } else {
    $SecurePassword = New-Object System.Security.SecureString
  }
  $Credential = New-Object System.Management.Automation.PSCredential $Username, $SecurePassword

  return $Credential
}

function Invoke-EnsureInitializedUser(
  [System.Management.Automation.PSCredential] $Credential
) {
  $Username = $Credential.UserName
  $HomePath = Join-Path "/Users" $Username

  if (Test-Path $HomePath) {
    return
  }

  $installKioskAppAsKioskUserParams = @{
    FilePath = "powershell.exe";
    Credential = $Credential;
    WorkingDirectory = $RootDirPath;
  }
  Start-Process @installKioskAppAsKioskUserParams -Wait `
    -ArgumentList "Write-Host 'Initializing user'"
}

[string] $RootDirPath = $env:ContentRootFolder;
if (-not $RootDirPath) {
  throw "Missing ContentRootFolder variable"
}
[string] $KioskFrontendInstallPath = $env:KioskFrontendInstallPath;
if (-not $KioskFrontendInstallPath) {
  throw "Missing KioskFrontendDirPath variable"
}
[string] $7ZipExecutableDirPath = $env:7ZipExecutableDirPath;
if (-not $7ZipExecutableDirPath) {
  throw "Missing ContentRootFolder variable"
}
[string] $KioskFrontendArchivePath = $env:KioskFrontendArchivePath;
if (-not $KioskFrontendArchivePath) {
  throw "Missing ContentRootFolder variable"
}
[string] $KioskUsername = $env:KioskUsername;
if (-not $KioskUsername) {
  throw "Missing KioskUsername variable"
}
[string] $KioskServerBinaryPath = $env:KioskServerBinaryRootPath;
if (-not $KioskServerBinaryPath) {
  throw "Missing KioskServerBinaryPath variable"
}
[string] $NodeExecutablePath = $env:NodeExecutablePath;
if (-not $NodeExecutablePath) {
  throw "Missing NodeExecutablePath variable"
}
[string] $WebViewArchivePath = $env:WebViewArchivePath;
if (-not $WebViewArchivePath) {
  throw "Missing WebViewArchivePath variable"
}
[string] $WebViewInstallPath = $env:WebViewInstallPath;
if (-not $WebViewInstallPath) {
  throw "Missing WebViewInstallPath variable"
}

Set-AllowAllTrustedApps
Invoke-EnsureAdministratorsGroup
Clear-AssignedAccess

$Credential = Get-UserCredentialsFromEnvironment
Invoke-EnsureInitializedUser $Credential
Invoke-LogOffuser $Credential.UserName

[string] $CurrentSetupRootPath = Join-Path $PSScriptRoot ".."

[string] $7ZipPath = $RootDirPath | Join-Path -ChildPath $7ZipExecutableDirPath | Join-Path -ChildPath "7z.exe"
[string] $AbsoluteKioskFrontendInstallPath = Join-Path $RootDirPath $KioskFrontendInstallPath
[string] $AbsoluteKioskFrontendArchivePath = Join-Path $CurrentSetupRootPath $KioskFrontendArchivePath
[string] $AbsoluteKioskFrontendExecPath = Join-Path $AbsoluteKioskFrontendInstallPath "SignageOS.KioskApp.exe"

New-Item -ItemType Directory -Path $AbsoluteKioskFrontendInstallPath -Force | Out-Null
& $7ZipPath x $AbsoluteKioskFrontendArchivePath -o"$AbsoluteKioskFrontendInstallPath" -r -y | Out-Null
[Environment]::SetEnvironmentVariable("KIOSK", "$AbsoluteKioskFrontendExecPath", [System.EnvironmentVariableTarget]::Machine)

[string] $AbsoluteWebViewArchivePath = Join-Path $CurrentSetupRootPath $WebViewArchivePath
[string] $AbsoluteWebViewInstallPath = Join-Path $RootDirPath $WebViewInstallPath
New-Item -ItemType Directory -Path $AbsoluteWebViewInstallPath -Force | Out-Null
Remove-Item -Path "$AbsoluteWebViewInstallPath/*" -Recurse -Force | Out-Null
$expandProcess = Start-Process `
  -FilePath "expand.exe" `
  -ArgumentList "-F:*", """$AbsoluteWebViewArchivePath""", """$AbsoluteWebViewInstallPath""" `
  -Wait `
  -PassThru

if ($expandProcess.ExitCode -ne 0) {
  throw "Failed to expand WebView archive, exit code $($expandProcess.ExitCode)"
}

[string] $AbsoluteWebViewExtractedInstallPath = Resolve-Path $(Join-Path $AbsoluteWebViewInstallPath "Microsoft.WebView2.*")
Move-Item -Path "$AbsoluteWebViewExtractedInstallPath\*" -Destination $AbsoluteWebViewInstallPath -Force | Out-Null
[Environment]::SetEnvironmentVariable("KIOSK_WEBVIEW", "$AbsoluteWebViewInstallPath", [System.EnvironmentVariableTarget]::Machine)

[string] $AbsoluteNodeExecutablePath = Join-Path $RootDirPath $NodeExecutablePath
[string] $AbsoluteKioskServerBinaryPath = Join-Path $RootDirPath $KioskServerBinaryPath
[string] $DesktopServerPath = Join-Path $AbsoluteKioskServerBinaryPath "dist/desktopServer/desktopServer.js"
[Environment]::SetEnvironmentVariable("DATADIR", "$AbsoluteKioskServerBinaryPath", [System.EnvironmentVariableTarget]::Machine)

$Shell = $AbsoluteNodeExecutablePath, $DesktopServerPath -join ' '
Set-UserShell -User $KioskUsername -Shell $Shell
