"use strict";
var __assign = (this && this.__assign) || function () {
    __assign = Object.assign || function(t) {
        for (var s, i = 1, n = arguments.length; i < n; i++) {
            s = arguments[i];
            for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p))
                t[p] = s[p];
        }
        return t;
    };
    return __assign.apply(this, arguments);
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.createSocket = createSocket;
exports.createAutoReconnectingSocket = createAutoReconnectingSocket;
var url_1 = require("url");
var ISocket_1 = require("../ISocket");
var events_1 = require("events");
var generator_1 = require("../../../Hash/generator");
var AutoReconnectingWSSocket_1 = __importDefault(require("./AutoReconnectingWSSocket"));
var isomorphic_ws_1 = __importDefault(require("isomorphic-ws"));
var debug_1 = __importDefault(require("debug"));
var debug = (0, debug_1.default)('@signageos/lib:WebSocket:Client:WS:createWSSocket');
var DEFAULT_PINGPONG_TIMEOUT = 30e3;
var defaultOptions = {
    enablePingPong: true,
    pingPongTimeout: DEFAULT_PINGPONG_TIMEOUT,
};
function createSocket(socketUri, onConnected, onDisconnected, onError, options) {
    if (options === void 0) { options = defaultOptions; }
    options = __assign(__assign({}, defaultOptions), options);
    var socketParsedUrl = (0, url_1.parse)(socketUri);
    if (socketParsedUrl.protocol === 'https:') {
        socketParsedUrl.protocol = 'wss:';
    }
    if (socketParsedUrl.protocol === 'http:') {
        socketParsedUrl.protocol = 'ws:';
    }
    var wsSocketUri = (0, url_1.format)(socketParsedUrl);
    var messageEmitter = new events_1.EventEmitter();
    var responseEmitter = new events_1.EventEmitter();
    var ws = new isomorphic_ws_1.default(wsSocketUri);
    var isAlive = true;
    function onMessage(messageEvent) {
        debug('message', messageEvent.data);
        var message = JSON.parse(messageEvent.data);
        switch (message.type) {
            case 'response':
                responseEmitter.emit(message.responseUid);
                break;
            case 'request':
                messageEmitter.emit(message.event, message.payload);
                if (message.responseUid) {
                    var responseMessage = { type: 'response', responseUid: message.responseUid };
                    if (ws.readyState === isomorphic_ws_1.default.OPEN) {
                        ws.send(JSON.stringify(responseMessage));
                    }
                    else {
                        debug('Socket is not open. Unconfirmed message', responseMessage, message);
                        var error = new ISocket_1.UnconfirmedMessageError(responseMessage, message, new Date());
                        onError(error);
                    }
                }
                break;
            case 'pong':
                isAlive = true;
                break;
            default:
        }
    }
    ws.addEventListener('message', onMessage);
    ws.addEventListener('error', onError);
    ws.addEventListener('open', onConnected);
    ws.addEventListener('close', onDisconnected);
    if (options.enablePingPong) {
        var pingPongInterval_1 = null;
        ws.addEventListener('open', function () {
            pingPongInterval_1 = setInterval(function () {
                if (!isAlive) {
                    debug('server not responding, closing websocket');
                    clearInterval(pingPongInterval_1);
                    ws.close(3000, 'Server is not responding');
                }
                else if (ws.readyState === isomorphic_ws_1.default.OPEN) {
                    debug('send ping');
                    var pingMessage = { type: 'ping' };
                    ws.send(JSON.stringify(pingMessage));
                    isAlive = false;
                }
            }, options.pingPongTimeout);
        });
        ws.addEventListener('close', function () {
            if (pingPongInterval_1) {
                clearInterval(pingPongInterval_1);
            }
        });
    }
    var socket = {
        on: function (event, listener) {
            messageEmitter.on(event, listener);
        },
        once: function (event, listener) {
            messageEmitter.once(event, listener);
        },
        emit: function (event, payload, callback) {
            var message = { type: 'request', event: event, payload: payload };
            if (callback) {
                message.responseUid = (0, generator_1.generateUniqueHash)();
                responseEmitter.once(message.responseUid, function () { return callback(); });
            }
            if (ws.readyState === isomorphic_ws_1.default.OPEN) {
                ws.send(JSON.stringify(message));
            }
            else {
                debug('Socket is not open. Undelivered message', message);
                var error = new ISocket_1.UndeliveredEmitError(message, new Date());
                onError(error);
                throw error;
            }
        },
        removeListener: function (event, listener) {
            messageEmitter.removeListener(event, listener);
        },
        removeAllListeners: function () {
            ws.removeEventListener('message', onMessage);
            ws.removeEventListener('error', onError);
            ws.removeEventListener('open', onConnected);
            ws.removeEventListener('close', onDisconnected);
            messageEmitter.removeAllListeners();
        },
        close: function () {
            if (ws.readyState !== isomorphic_ws_1.default.CLOSED && ws.readyState !== isomorphic_ws_1.default.CLOSING) {
                ws.close(1000, 'Closed by client');
            }
            else {
                debug('Closing disconnected socket is not wanted');
            }
        },
    };
    return socket;
}
function createAutoReconnectingSocket(socketUri, onConnected, onDisconnected, onError, options, initialReconnectTimeout, maxReconnectTimeout) {
    var socketFactory = function (onSocketConnected, onSocketDisconnected, onSocketError) {
        return createSocket(socketUri, onSocketConnected, onSocketDisconnected, onSocketError, options);
    };
    var autoReconnectingSocket = new AutoReconnectingWSSocket_1.default(socketFactory, initialReconnectTimeout, maxReconnectTimeout);
    autoReconnectingSocket.onConnected(onConnected);
    autoReconnectingSocket.onDisconnected(onDisconnected);
    autoReconnectingSocket.onError(onError);
    return autoReconnectingSocket;
}
//# sourceMappingURL=createWSSocket.js.map