"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.accessLogMiddleware = accessLogMiddleware;
var on_finished_1 = __importDefault(require("on-finished"));
var basic_auth_1 = __importDefault(require("basic-auth"));
function accessLogMiddleware(logCallback) {
    return function (req, res, next) {
        // record request start time and use it to calculate total time in the end
        req._startAt = process.hrtime.bigint();
        (0, on_finished_1.default)(res, function (_err, resFinished) {
            var accessLog = createAccessLogMessage(req, resFinished);
            logCallback(accessLog, req, resFinished);
        });
        next();
    };
}
/**
 * @returns A log message in format based on the Nginx combined format
 * @see https://nginx.org/en/docs/http/ngx_http_log_module.html
 */
function createAccessLogMessage(req, res) {
    var _a, _b, _c;
    return {
        method: req.method,
        path: (_a = req.originalUrl) !== null && _a !== void 0 ? _a : req.url,
        status: res.statusCode,
        remoteAddress: getRemoteAddress(req),
        userAgent: req.get('User-Agent'),
        requestBytes: parseInt((_b = req.get('Content-Length')) !== null && _b !== void 0 ? _b : '0'),
        responseBytes: parseInt((_c = res.get('Content-Length')) !== null && _c !== void 0 ? _c : '0'),
        headers: getHeaders(req),
        referrer: getReferrer(req),
        remoteUser: getRemoteUser(req),
        tookSeconds: getTotalTime(req),
    };
}
function getRemoteAddress(req) {
    var _a, _b;
    var xForwardedFor = req.get('X-Forwarded-For');
    // if the request is coming through a proxy, the original client IP will be in the X-Forwarded-For header
    // https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/X-Forwarded-For
    if (xForwardedFor) {
        return xForwardedFor.split(',')[0].trim();
    }
    return (_b = (_a = req.ip) !== null && _a !== void 0 ? _a : req.connection.remoteAddress) !== null && _b !== void 0 ? _b : undefined;
}
function getHeaders(req) {
    var HEADERS_TO_LOG = ['cloudfront-viewer-country', 'cloudfront-viewer-asn'];
    var headers = {};
    for (var _i = 0, HEADERS_TO_LOG_1 = HEADERS_TO_LOG; _i < HEADERS_TO_LOG_1.length; _i++) {
        var header = HEADERS_TO_LOG_1[_i];
        var value = req.get(header);
        if (value) {
            headers[header] = value;
        }
    }
    return headers;
}
function getReferrer(req) {
    var _a;
    // "Referer" is actually mis-spelled but it's the standard according to the spec
    // "Referrer" is correct spelling and sometimes used instead
    // https://en.wikipedia.org/wiki/HTTP_referer
    return (_a = req.get('Referer')) !== null && _a !== void 0 ? _a : req.get('Referrer');
}
function getRemoteUser(req) {
    var _a;
    var credentials = (0, basic_auth_1.default)(req);
    return (_a = credentials === null || credentials === void 0 ? void 0 : credentials.name) !== null && _a !== void 0 ? _a : undefined;
}
/**
 * Returns the time between the request coming into the middleware and
 * when the response has finished being written out to the connection, in seconds.
 *
 * @return The time in seconds or 0 on error.
 */
function getTotalTime(req) {
    if (!req._startAt) {
        // missing request start time
        return 0;
    }
    // time elapsed from request start
    var ns = process.hrtime.bigint() - req._startAt;
    // convert to seconds
    return Number(ns / BigInt(1e9));
}
//# sourceMappingURL=accessLogMiddleware.js.map