import { Feature, Point } from 'geojson';
import { ICreatable, IUpdatable, IArchivable } from '../../Lib/Model/model';
import { IOrganization } from '../Organization/organizationModel';
import { IPaginator, PaginatorType } from '../../MongoDB/paginatorHelper';
import { PaginationAndSorting } from '../../Lib/Pagination/paginationTypes';
import { IOrganizationTag, OrganizationTagFamilyTreeUids } from './../Organization/Tag/organizationTagModel';
import { LocationMarker } from '@signageos/common-types/dist/Location/location';
import { IModelCreatable } from '../../Lib/Model/IModelCreatable';
import { IModelUpdatable } from '../../Lib/Model/IModelUpdatable';
import { IModelDeletable } from '../../Lib/Model/IModelDeletable';
import { IModelReadable } from '../../Lib/Model/IModelReadable';
export declare type LocationSortIndices = 'createdAt' | 'updatedAt';
export declare enum LocationStatus {
    ACTIVE = 0,
    ARCHIVED = 1,
    ALL = 2
}
export declare type PropertyKey = 'city' | 'country' | 'state' | 'tagUids';
export interface ILocation extends ICreatable, IUpdatable, IArchivable {
    uid: string;
    name: string;
    feature: Feature<Point>;
    organizationUid: string;
    city: string;
    countryCode: string;
    customId?: string | null;
    attachments: string[];
    description?: string | null;
    tagUids?: string[];
    mapMarker?: LocationMarker | null;
}
export declare type LocationIdentification = Pick<ILocation, 'uid'>;
/**
 * @deprecated Don't use this interface and don't rely on deviceCount property.
 * It doesn´t consider device filter from company network if exists so it could
 * return wrong number of devices in that case.
 */
export interface ILocationWithDeviceNumber extends ILocation {
    deviceNumber: number;
}
export interface ILocationPropertyFilter {
    uids?: string[];
    organizationUids?: string[];
    tagUids?: string[];
    /**
     * Each array inside array should represent group of tag and its children.
     * This argument allows to include all locations that have at least one tag from each family tree.
     */
    tagsWithChildrenUids?: OrganizationTagFamilyTreeUids[];
    name?: string;
    cities?: string[];
    states?: string[];
    countries?: string[];
}
export interface ILocationModelWrite<TLocation extends ILocation> extends IModelCreatable<TLocation>, IModelUpdatable<TLocation>, IModelDeletable<TLocation> {
    archive(locationRow: ILocation, archivedAt: Date): Promise<void>;
    unarchive(locationRow: ILocation, unarchivatedAt: Date): Promise<void>;
    assignTag(locationRow: TLocation, organizationTagRow: IOrganizationTag): Promise<void>;
    removeTag(locationRow: ILocation, organizationTagRow: IOrganizationTag): Promise<void>;
}
export interface ILocationModelRead<TLocation extends ILocation, TLocationWithDeviceNumber extends ILocationWithDeviceNumber> extends IModelReadable<TLocation | TLocationWithDeviceNumber> {
    fetchByUids(uids: string[], status?: LocationStatus): Promise<TLocation[] | null>;
    fetchByUid<D extends boolean | undefined>(uid: string, 
    /**
     * @deprecated Don't use this method to return locations with deviceNumber property.
     * It doesn´t consider device filter from company network if exists so it could
     * return wrong number of devices in that case.
     */
    withDeviceNumber?: D): Promise<D extends true ? TLocationWithDeviceNumber | null : TLocation | null>;
    /** @deprecated Use fetchListByPropertyFilterV2 instead */
    fetchListByOrganizations(organizations: IOrganization[], pagination?: IPaginator<LocationSortIndices, PaginatorType>, status?: LocationStatus): Promise<ILocation[]>;
    /**
     * @deprecated Use fetchListByPropertyFilterV2 instead
     * This method can also return list of location with deviceNumber property.
     * It doesn´t consider device filter from company network if exists so it could
     * return wrong number of devices in that case.
     */
    fetchListByPropertyFilter<D extends boolean>(filter: ILocationPropertyFilter, 
    /** @deprecated Don't use this method to return locations with deviceNumber property. */
    withDeviceNumber: D, pagination?: IPaginator<LocationSortIndices, PaginatorType>, status?: LocationStatus): Promise<D extends true ? TLocationWithDeviceNumber[] : TLocation[]>;
    fetchListByPropertyFilterV2({ filter, paginationAndSorting, status, }: {
        filter: ILocationPropertyFilter;
        paginationAndSorting?: PaginationAndSorting<LocationSortIndices>;
        status?: LocationStatus;
    }): Promise<ILocation[]>;
    fetchTotalCountByPropertyFilter(filter: ILocationPropertyFilter, status?: LocationStatus): Promise<number>;
    fetchPropertyValuesByPropertyFilter({ filter, propertyKey, status, }: {
        filter: ILocationPropertyFilter;
        propertyKey: PropertyKey;
        status?: LocationStatus;
    }): Promise<string[]>;
    fetchPropertyValuesCountByPropertyFilter({ filter, propertyKey, status, }: {
        filter: ILocationPropertyFilter;
        propertyKey: PropertyKey;
        status?: LocationStatus;
    }): Promise<number>;
}
//# sourceMappingURL=locationModel.d.ts.map