/// <reference types="lib.es2017.observable" />
import { IDevice } from '../deviceModel';
import { IPackageVersion } from '../../Package/packageVersionModel';
export interface IDeviceInstalledPackage {
    uid: string;
    deviceIdentityHash: string;
    /**
     * Contains unique hash (usually checksum of package file) which is unique for every build
     * E.g.:
     * Android - apk file sha256
     * TODO other platforms
     */
    buildHash: string;
    specs: IAndroidSpecs;
    createdAt: Date;
}
/**
 * The specifications of installed android package (APK file).
 * It contains extra information specific only for the android platform (applicationType).
 * Some of those can be used as a source for buildHash computation.
 */
export interface IAndroidSpecs {
    /** `packageName` as defined by the APK manifest, i.e. com.example.app */
    packageName: string;
    /** `versionCode` as defined by the APK manifest, i.e. 200050 */
    versionCode: number;
    /** SHA-256 hash of the APK file. */
    apkSha256: string;
    /**
     * `versionName` as defined by the APK manifest, i.e. 2.3.0.
     * Older Core Apps did not have this field, so it can be omitted.
     */
    versionName?: string;
    /**
     * SHA-256 hash of the APK signature.
     * Older Core Apps did not have this field, so it can be omitted.
     */
    signatureSha256?: string;
    flags?: Array<'SYSTEM' | 'DEBUGGABLE' | 'UPDATED_SYSTEM_APP' | 'TEST_ONLY' | 'EXTERNAL_STORAGE'>;
    targetSdkVersion?: number;
}
export declare type IDeviceInstalledPackageModel<T extends IDeviceInstalledPackage> = IDeviceInstalledPackageModelRead<T> & IDeviceInstalledPackageModelWrite & IDeviceInstalledPackageModelObservable<T>;
export interface IDeviceInstalledPackageModelRead<T extends IDeviceInstalledPackage> {
    fetchListByDevice(device: IDevice): Promise<T[]>;
    fetchByDeviceAndBuildHash(device: IDevice, buildHash: string): Promise<T | null>;
    fetchListByDeviceAndPackageVersions(device: IDevice, packageVersions: IPackageVersion[]): Promise<T[]>;
}
export interface IDeviceInstalledPackageModelWrite {
    create(uid: string, device: IDevice, buildHash: string, specs: IAndroidSpecs, createdAt: Date): Promise<void>;
    delete(device: IDevice, buildHash: IDeviceInstalledPackage['buildHash']): Promise<void>;
}
export interface IDeviceInstalledPackageModelObservable<T extends IDeviceInstalledPackage> {
    observeListByDevice(device: IDevice): Promise<Observable<T[], Error>>;
}
//# sourceMappingURL=deviceInstalledPackageModel.d.ts.map