/// <reference types="lib.es2017.observable" />
import { IDevice } from '../deviceModel';
export declare type SocketName = 'front' | 'management' | 'service';
export interface IDeviceDisconnection {
    duid: string;
    socketName: SocketName | null;
    /** If not provided, all connections are disconnected not excluding this one */
    excludeConnectionUid?: string;
}
export interface IDeviceConnection {
    uid: string;
    deviceUid: string;
    serverInstanceUid: string;
    socketName: SocketName;
    createdAt: Date;
    organizationUid?: string | null;
}
export declare type IRemovableDeviceConnection = Pick<IDeviceConnection, 'uid' | 'deviceUid' | 'organizationUid' | 'socketName'>;
export declare type IDeviceForCreate = Pick<IDevice, 'uid' | 'organizationUid'>;
export interface IDeviceConnectionIndex {
    [deviceUid: string]: IDeviceConnection[];
}
export interface IDeviceConnectionCountIndex {
    [deviceUid: string]: number;
}
export interface IOrganizationDeviceConnectionCountIndex {
    [organizationUid: string]: IDeviceConnectionCountIndex;
}
export interface IDeviceConnectionModel {
    fetchByUid(uid: string): Promise<IDeviceConnection | null>;
    fetchListByDevice(device: IDevice): Promise<IDeviceConnection[]>;
    fetchListByDeviceUid(deviceUid: string): Promise<IDeviceConnection[]>;
    fetchIndexByDeviceUids(deviceUids: string[]): Promise<IDeviceConnectionIndex>;
    fetchCountIndexByOrganizationUids(organizationUids: string[]): Promise<IOrganizationDeviceConnectionCountIndex>;
    updateIndexes(): Promise<void>;
    create(uid: string, device: IDeviceForCreate, serverInstanceUid: string, createdAt: Date, socketName: SocketName): Promise<void>;
    remove(deviceConnectionRow: IRemovableDeviceConnection): Promise<void>;
    updateDeviceOrganization(device: IDevice, organizationUid: string | null): Promise<void>;
    /**
     * Initiate disconnection of one specific socket of device or all it's sockets.
     * If socketName is null, all device sockets are disconnected.
     * This is useful for faster reconnect of device from box/api
     * and when device is connected to server again and old TCP connection did not timed out yet.
     * It can happen when device is disconnected unexpectedly.
     * @param excludeConnectionUid is optional to prevent disconnection of exactly this connection
     */
    disconnect(duid: string, socketName: SocketName | null, excludeConnectionUid?: string): Promise<void>;
    /**
     * Observes all requests for disconnect device from server initiated by disconnect method.
     * It's idempotent action, so only if specific connection exists, it's forcedly disconnected.
     */
    observeDisconnections(): Observable<IDeviceDisconnection, Error>;
}
//# sourceMappingURL=deviceConnectionModel.d.ts.map