"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.observeWithInitialData = exports.observeSimple = exports.observe = void 0;
const mongodb_1 = require("mongodb");
const Debug = require("debug");
const MemoryObservationManager_1 = require("./MemoryObservationManager");
const datetime_1 = require("./datetime");
const SimpleStreamWatcher_1 = require("./Watcher/SimpleStreamWatcher");
const debug = Debug('@signageos/user-domain-model:MongoDB:observation');
function observe(collection, filter, options = {}) {
    return __awaiter(this, void 0, void 0, function* () {
        const startAtOperationTime = options.startAtOperationTime || (yield (0, datetime_1.getCurrentTimestamp)(collection.conn.db));
        // TODO used internal API collection.conn.db
        const recoverIntervalMs = options.recoverIntervalMs || 60e3;
        const subscriptionItems = [];
        let lastChangeClusterTime = new mongodb_1.Timestamp(startAtOperationTime.subtract(mongodb_1.Timestamp.fromBits(1, 0)));
        let observable = yield observeSimple(collection, filter, Object.assign(Object.assign({}, options), { startAtOperationTime }));
        let recoveryIntervalHandler;
        function forwardSubscriptionToCurrentObservable(observer) {
            return observable.subscribe((changes) => {
                lastChangeClusterTime = changes.clusterTime;
                observer.next(changes);
            }, (error) => observer.error(error), () => observer.complete());
        }
        function startRecoverying() {
            recoveryIntervalHandler = setInterval(() => __awaiter(this, void 0, void 0, function* () {
                if (subscriptionItems.length > 0) {
                    observable = yield observeSimple(collection, filter, Object.assign(Object.assign({}, options), { 
                        // TODO ids can differs (no cluster time ids currently)
                        startAtOperationTime: new mongodb_1.Timestamp(lastChangeClusterTime.add(mongodb_1.Timestamp.fromBits(1, 0))) }));
                    for (const subscriptionItem of subscriptionItems) {
                        subscriptionItem.subscription.unsubscribe();
                        subscriptionItem.subscription = forwardSubscriptionToCurrentObservable(subscriptionItem.observer);
                    }
                }
            }), recoverIntervalMs);
        }
        return new Observable((observer) => {
            debug('Subscribe recoverable observable', filter);
            const subscriptionItem = {
                subscription: forwardSubscriptionToCurrentObservable(observer),
                observer,
            };
            subscriptionItems.push(subscriptionItem);
            if (subscriptionItems.length === 1) {
                startRecoverying();
            }
            return () => {
                debug('Unsubscribe recoverable observable', filter);
                subscriptionItems.splice(subscriptionItems.indexOf(subscriptionItem), 1);
                subscriptionItem.subscription.unsubscribe();
                if (subscriptionItems.length === 0) {
                    clearTimeout(recoveryIntervalHandler);
                }
            };
        });
    });
}
exports.observe = observe;
function observeSimple(collection, filter, options = {}) {
    return __awaiter(this, void 0, void 0, function* () {
        const streamWatcher = options.streamWatcher || new SimpleStreamWatcher_1.default();
        const initialWatchOptions = {
            session: options.session,
            startAtOperationTime: options.startAtOperationTime || (yield (0, datetime_1.getCurrentTimestamp)(collection.conn.db)),
            // TODO used internal API collection.conn.db
        };
        let lastChangeClusterTime;
        const ids = yield getObjectIds(collection, filter);
        debug('observation initial ids', ids, initialWatchOptions.startAtOperationTime, filter);
        const changeStreamFilter = getChangeStreamFilter(filter);
        return new Observable((observer) => {
            const subscribtionWatchOptions = Object.assign(Object.assign({}, initialWatchOptions), { startAtOperationTime: lastChangeClusterTime
                    ? new mongodb_1.Timestamp(lastChangeClusterTime.add(mongodb_1.Timestamp.fromBits(1, 0)))
                    : initialWatchOptions.startAtOperationTime });
            const updateAndNewStream = streamWatcher.watchUpserts(collection, changeStreamFilter, subscribtionWatchOptions);
            const deleteAndRemoveStream = streamWatcher.watchDeleteAndRemove(collection, changeStreamFilter, ids, subscribtionWatchOptions);
            const observationManager = new MemoryObservationManager_1.default(ids, (value) => {
                lastChangeClusterTime = value.clusterTime;
                observer.next(value);
            }, (newId, clusterTime) => {
                const newWatchOptions = Object.assign(Object.assign({}, initialWatchOptions), { startAtOperationTime: new mongodb_1.Timestamp(clusterTime.add(mongodb_1.Timestamp.fromBits(1, 0))) });
                const newDeleteAndRemoveStream = streamWatcher.watchDeleteAndRemove(collection, changeStreamFilter, [newId], newWatchOptions);
                newDeleteAndRemoveStream.on('change', observationManager.deleteOrRemoveListener);
                return () => __awaiter(this, void 0, void 0, function* () {
                    newDeleteAndRemoveStream.removeListener('change', observationManager.deleteOrRemoveListener);
                    yield newDeleteAndRemoveStream.close();
                });
            });
            updateAndNewStream.on('change', observationManager.updateAndNewListener);
            deleteAndRemoveStream.on('change', observationManager.deleteOrRemoveListener);
            return () => {
                updateAndNewStream.removeListener('change', observationManager.updateAndNewListener);
                deleteAndRemoveStream.removeListener('change', observationManager.deleteOrRemoveListener);
                Promise.all([updateAndNewStream.close(), deleteAndRemoveStream.close(), observationManager.drain()])
                    .then(() => debug(`Observable subscription closed`))
                    .catch((error) => {
                    throw error;
                });
            };
        });
    });
}
exports.observeSimple = observeSimple;
// Legacy 'changefeed' capability inherited from rethinkdb
// we need it for backward compatibility
function observeWithInitialData(collection, filter, options = {}, initialData) {
    return __awaiter(this, void 0, void 0, function* () {
        const observable = yield observe(collection, filter, options);
        let dbEntity = null;
        if (initialData) {
            dbEntity = yield collection.findOne(filter);
        }
        return new Observable((observer) => {
            if (initialData) {
                observer.next({
                    new_val: dbEntity,
                    old_val: dbEntity,
                    clusterTime: mongodb_1.Timestamp.fromBits(0, Math.floor(new Date().getTime() / 1000)),
                });
            }
            return observable.subscribe((changes) => {
                observer.next(changes);
            }, (error) => observer.error(error), () => observer.complete());
        });
    });
}
exports.observeWithInitialData = observeWithInitialData;
function getChangeStreamFilter(filter) {
    return Object.keys(filter).reduce((csf, key) => {
        if (key.search(/\$or/gi) >= 0) {
            return Object.assign(Object.assign({}, csf), { ['$or']: filter[key].map((value) => {
                    return getChangeStreamFilter(value);
                }) });
        }
        return Object.assign(Object.assign({}, csf), { [`fullDocument.${key}`]: filter[key] });
    }, {});
}
function getObjectIds(collection, filter) {
    return __awaiter(this, void 0, void 0, function* () {
        const idObjects = (yield collection.find(filter, { projection: { _id: 1 } }).toArray());
        const ids = idObjects.map((idObject) => `${idObject._id}`);
        return ids;
    });
}
//# sourceMappingURL=observation.js.map