"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const _ = require("lodash");
const Debug = require("debug");
const mongodb_1 = require("mongodb");
const CombinedChangeStream_1 = require("../CombinedChangeStream");
const conditions_1 = require("../conditions");
const ProxyChangeStream_1 = require("../ProxyChangeStream");
const mingo = require('mingo');
const debug = Debug('@signageos/user-domain-model:MongoDB:Watcher:IntervalStreamWatcher');
class IntervalStreamWatcher {
    constructor(intervalMs) {
        this.collectionStatesMap = {};
        this.start(intervalMs);
    }
    watchDeleteAndRemove(collection, changeStreamFilter, removableAndDeletableIds, options) {
        const deleteStream = this.watchDeletions(collection, removableAndDeletableIds, options);
        const removeStream = this.watchRemoves(collection, changeStreamFilter, removableAndDeletableIds, options);
        return new CombinedChangeStream_1.default(deleteStream, removeStream);
    }
    watchUpserts(collection, changeStreamFilter, options) {
        const streamChangeFilterQuery = Object.assign(Object.assign({}, changeStreamFilter), { operationType: { $in: ['insert', 'replace', 'update'] } });
        return this.setupCollectionFilter(collection, streamChangeFilterQuery, options);
    }
    watchDeletions(collection, removableAndDeletableIds, options) {
        const removableAndDeletableObjectIds = removableAndDeletableIds.map((id) => new mongodb_1.ObjectId(id));
        const streamChangeFilterQuery = {
            'documentKey._id': { $in: removableAndDeletableObjectIds },
            operationType: 'delete',
        };
        return this.setupCollectionFilter(collection, streamChangeFilterQuery, options);
    }
    watchRemoves(collection, changeStreamFilter, removableAndDeletableIds, options) {
        const removableAndDeletableObjectIds = removableAndDeletableIds.map((id) => new mongodb_1.ObjectId(id));
        const changeStreamFilterReversed = (0, conditions_1.getReversedTopLevelCondition)(changeStreamFilter);
        const streamChangeFilterQuery = Object.assign(Object.assign({}, changeStreamFilterReversed), { 'documentKey._id': { $in: removableAndDeletableObjectIds }, operationType: { $in: ['replace', 'update'] } });
        return this.setupCollectionFilter(collection, streamChangeFilterQuery, options);
    }
    drain() {
        return __awaiter(this, void 0, void 0, function* () {
            clearInterval(this.intervalHandler);
            debug('Draining', this.collectionStatesMap);
            yield Promise.all(Object.keys(this.collectionStatesMap).map((collectionName) => __awaiter(this, void 0, void 0, function* () {
                yield Promise.all(this.collectionStatesMap[collectionName].filters.map((filter) => __awaiter(this, void 0, void 0, function* () {
                    yield filter.changeStream.close();
                })));
                const currentChangeStream = this.collectionStatesMap[collectionName].currentChangeStream;
                if (currentChangeStream) {
                    yield currentChangeStream.close();
                }
            })));
        });
    }
    setupCollectionFilter(collection, streamChangeFilterQuery, options) {
        const startAtOperationTime = options.startAtOperationTime;
        const collectionState = this.ensureCollectionState(collection, startAtOperationTime);
        collectionState.changed = true;
        const filter = {
            streamChangeFilterQuery,
            startAtOperationTime,
            changeStream: new ProxyChangeStream_1.default(() => {
                collectionState.changed = true;
                this.removeFilterMutation(collectionState, filter);
                debug(`Collection ${collection.collectionName} has ${collectionState.filters.length} filters`);
                this.cleanupCollectionState(collection);
                return Promise.resolve();
            }),
        };
        this.addFilterMutation(collectionState, filter);
        debug(`Collection ${collection.collectionName} has ${collectionState.filters.length} filters`);
        this.warnPossibleMemoryLeak(collectionState, collection);
        return filter.changeStream;
    }
    removeFilterMutation(collectionState, filter) {
        collectionState.filters.splice(collectionState.filters.indexOf(filter), 1);
    }
    addFilterMutation(collectionState, filter) {
        collectionState.filters.push(filter);
    }
    start(intervalMs) {
        this.intervalHandler = setInterval(() => {
            for (let collectionName in this.collectionStatesMap) {
                this.processCollectionChanges(collectionName);
            }
        }, intervalMs);
    }
    processCollectionChanges(collectionName) {
        const collectionState = this.collectionStatesMap[collectionName];
        debug(`invoke interval`, collectionState.filters);
        if (collectionState.changed) {
            debug(`changed collection ${collectionName}`);
            collectionState.changed = false;
            const collection = collectionState.collection;
            const filters = collectionState.filters;
            const lastChangeClusterTime = collectionState.lastChangeClusterTime;
            this.closeAndRemoveChangeStream(collectionName);
            this.fillFiltersStartAtOperationTime(filters, lastChangeClusterTime);
            const startAtOperationTime = this.getTheLowestStartAtOperationTime(filters);
            debug(`The lowest cluster time`, collectionName, startAtOperationTime);
            const currentChangeStream = collection.watch([{ $match: this.getAnyFilterCondition(filters) }], this.getChangeStreamOptions(startAtOperationTime));
            currentChangeStream.on('change', (streamChange) => {
                collectionState.lastChangeClusterTime = streamChange.clusterTime;
                this.handleStreamChange(filters, streamChange);
            });
            collectionState.currentChangeStream = currentChangeStream;
        }
    }
    handleStreamChange(filters, streamChange) {
        for (const filter of filters) {
            const mingoFilterQuery = new mingo.Query(filter.streamChangeFilterQuery);
            const isClusterTimeReached = typeof filter.startAtOperationTime === 'undefined' || streamChange.clusterTime.greaterThanOrEqual(filter.startAtOperationTime);
            debug(`Checking filter change`, filter, isClusterTimeReached, streamChange);
            if (isClusterTimeReached && mingoFilterQuery.test(streamChange)) {
                filter.changeStream.emit('change', streamChange);
                delete filter.startAtOperationTime;
            }
        }
    }
    fillFiltersStartAtOperationTime(filters, lastChangeClusterTime) {
        for (const filter of filters) {
            if (typeof filter.startAtOperationTime === 'undefined') {
                filter.startAtOperationTime = new mongodb_1.Timestamp(lastChangeClusterTime.add(mongodb_1.Timestamp.fromBits(1, 0)));
            }
        }
    }
    closeAndRemoveChangeStream(collectionName) {
        const oldChangeStream = this.collectionStatesMap[collectionName].currentChangeStream;
        if (oldChangeStream) {
            oldChangeStream.removeAllListeners('change');
            oldChangeStream
                .close()
                .then(() => {
                debug(`Old change stream successfully closed`);
            })
                .catch((error) => {
                throw error;
            });
            delete this.collectionStatesMap[collectionName].currentChangeStream;
        }
    }
    getChangeStreamOptions(startAtOperationTime) {
        return {
            fullDocument: 'updateLookup',
            startAtOperationTime,
        };
    }
    cleanupCollectionState(collection) {
        if (this.collectionStatesMap[collection.collectionName].filters.length === 0) {
            const currentChangeStream = this.collectionStatesMap[collection.collectionName].currentChangeStream;
            delete this.collectionStatesMap[collection.collectionName];
            if (currentChangeStream) {
                currentChangeStream.removeAllListeners('change');
                return currentChangeStream.close(); // has to be after delete
            }
        }
    }
    ensureCollectionState(collection, startAtOperationTime) {
        if (typeof this.collectionStatesMap[collection.collectionName] === 'undefined') {
            this.collectionStatesMap[collection.collectionName] = this.createCollectionState(collection, startAtOperationTime);
        }
        return this.collectionStatesMap[collection.collectionName];
    }
    createCollectionState(collection, startAtOperationTime) {
        return {
            collection,
            changed: false,
            filters: [],
            lastChangeClusterTime: new mongodb_1.Timestamp(startAtOperationTime.subtract(mongodb_1.Timestamp.fromBits(1, 0))),
        };
    }
    getTheLowestStartAtOperationTime(filters) {
        return _.minBy(filters.map((filter) => filter.startAtOperationTime), (timestamp) => timestamp.toNumber());
    }
    getAnyFilterCondition(filters) {
        return {
            $or: filters.map((filter) => filter.streamChangeFilterQuery),
        };
    }
    warnPossibleMemoryLeak(collectionState, collection) {
        if (collectionState.filters.length > 20) {
            console.warn(`Too many filters for collection ${collection.collectionName}. Possible memory or CPU leak`, collectionState.filters.length);
        }
    }
}
exports.default = IntervalStreamWatcher;
//# sourceMappingURL=IntervalStreamWatcher.js.map