"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.createSystemLogModel = exports.prepareSystemLogTable = exports.systemLogCollection = void 0;
const escape_1 = require("../../MongoDB/escape");
const collections_1 = require("../Lib/collections");
const utils_1 = require("../Lib/utils");
const systemLog_utils_1 = require("../../Schema/SystemLog/systemLog.utils");
const systemLogCollection = (conn) => conn.connection.collection(collections_1.Collection.SystemLog, {
    writeConcern: {
        w: 1,
    },
});
exports.systemLogCollection = systemLogCollection;
const prepareSystemLogTable = (conn) => __awaiter(void 0, void 0, void 0, function* () {
    yield (0, exports.systemLogCollection)(conn).createIndex({ id: 1 }, { name: 'id', unique: true, background: true });
    yield (0, exports.systemLogCollection)(conn).createIndex({ deviceIdentityHash: 1 }, { name: 'deviceIdentityHash', background: true });
    yield (0, exports.systemLogCollection)(conn).createIndex({ receivedAt: 1 }, { name: 'receivedAt', background: true, expireAfterSeconds: 30 * 24 * 60 * 60 });
    yield (0, exports.systemLogCollection)(conn).createIndex({ recordedAt: 1 }, { name: 'recordedAt', background: true, expireAfterSeconds: 30 * 24 * 60 * 60 });
    yield (0, exports.systemLogCollection)(conn).createIndex({ 'payload.type': 1 }, { name: 'payload_type', background: true });
    yield (0, exports.systemLogCollection)(conn).createIndex({ deviceIdentityHash: 1, receivedAt: 1 }, { name: 'deviceIdentityHash_receivedAt', background: true });
    yield (0, exports.systemLogCollection)(conn).createIndex({ deviceIdentityHash: 1, receivedAt: -1 }, { name: 'deviceIdentityHash_receivedAt-1', background: true });
    yield (0, exports.systemLogCollection)(conn).createIndex({ deviceIdentityHash: 1, recordedAt: 1 }, { name: 'deviceIdentityHash_recordedAt', background: true });
    yield (0, exports.systemLogCollection)(conn).createIndex({ deviceIdentityHash: 1, recordedAt: -1 }, { name: 'deviceIdentityHash_recordedAt-1', background: true });
    yield (0, exports.systemLogCollection)(conn).createIndex({ deviceIdentityHash: 1, appletUid: 1 }, { name: 'deviceIdentityHash_appletUid', background: true });
    yield (0, exports.systemLogCollection)(conn).createIndex({ deviceIdentityHash: 1, 'payload.type': 1 }, { name: 'deviceIdentityHash_payload_type', background: true });
    yield (0, exports.systemLogCollection)(conn).createIndex({ deviceIdentityHash: 1, 'payload.type': 1, receivedAt: -1 }, { name: 'deviceIdentityHash_payload_type_receivedAt', background: true });
    yield (0, exports.systemLogCollection)(conn).createIndex({ deviceIdentityHash: 1, 'payload.type': 1, recordedAt: -1 }, { name: 'deviceIdentityHash_payload_type_recordedAt', background: true });
});
exports.prepareSystemLogTable = prepareSystemLogTable;
function mapRow(row) {
    var _a;
    return Object.assign(Object.assign({}, row), { 
        // TODO inline migration of old data, remove once all data have "recordedAt" field
        recordedAt: (_a = row.recordedAt) !== null && _a !== void 0 ? _a : row.receivedAt, payload: (0, escape_1.unescapeObject)(row.payload) });
}
function getFilterByDeviceAndPropertyFilter(device, propertyFilter) {
    const filter = [];
    filter.push({ deviceIdentityHash: device.identityHash });
    if (propertyFilter.type !== undefined) {
        filter.push({ 'payload.type': { $regex: `^${propertyFilter.type}` } });
    }
    if (propertyFilter.excludeType !== undefined) {
        filter.push({ 'payload.type': { $regex: `^(?!${propertyFilter.excludeType})` } });
    }
    if (typeof propertyFilter.receivedSince !== 'undefined') {
        filter.push({ receivedAt: { $gte: propertyFilter.receivedSince } });
    }
    if (typeof propertyFilter.receivedUntil !== 'undefined') {
        filter.push({ receivedAt: { $lt: propertyFilter.receivedUntil } });
    }
    if (typeof propertyFilter.recordedSince !== 'undefined') {
        filter.push({ recordedAt: { $gte: propertyFilter.recordedSince } });
    }
    if (typeof propertyFilter.recordedUntil !== 'undefined') {
        filter.push({ recordedAt: { $lt: propertyFilter.recordedUntil } });
    }
    return filter;
}
const createSystemLogModel = (conn) => ({
    fetchById(id) {
        return __awaiter(this, void 0, void 0, function* () {
            const systemLog = yield (0, exports.systemLogCollection)(conn).findOne({ id });
            return systemLog ? mapRow(systemLog) : null;
        });
    },
    fetchListByDeviceAndPropertyFilter(device, propertyFilter, pagination, customSorting) {
        return __awaiter(this, void 0, void 0, function* () {
            const filter = getFilterByDeviceAndPropertyFilter(device, propertyFilter);
            const matcher = { $and: filter };
            const sorting = (0, systemLog_utils_1.getSorting)(customSorting);
            const pipeline = (0, utils_1.getPipeline)({ matcher, sorting, pagination });
            const rows = yield (0, exports.systemLogCollection)(conn).aggregate(pipeline, { session: conn.session }).toArray();
            return rows.map(mapRow);
        });
    },
    countByDeviceAndPropertyFilter(device, propertyFilter) {
        return __awaiter(this, void 0, void 0, function* () {
            const filter = getFilterByDeviceAndPropertyFilter(device, propertyFilter);
            // TODO remove as any
            // "as any" is a huge crime but I'm running out of time.
            // It started throwing TS error after upgrading mongoose but the tests are still passing
            let query = (0, exports.systemLogCollection)(conn).count({ $and: filter });
            return query;
        });
    },
    fetchUniqueTypesByDevice({ device, receivedSince, receivedUntil, recordedSince, recordedUntil }) {
        return __awaiter(this, void 0, void 0, function* () {
            const filter = [];
            filter.push({ deviceIdentityHash: device.identityHash });
            if (typeof receivedSince !== 'undefined') {
                filter.push({ receivedAt: { $gte: receivedSince } });
            }
            if (typeof receivedUntil !== 'undefined') {
                filter.push({ receivedAt: { $lt: receivedUntil } });
            }
            if (typeof recordedSince !== 'undefined') {
                filter.push({ recordedAt: { $gte: recordedSince } });
            }
            if (typeof recordedUntil !== 'undefined') {
                filter.push({ recordedAt: { $lt: recordedUntil } });
            }
            // TODO remove as any
            // "as any" is a huge crime but I'm running out of time.
            // It started throwing TS error after upgrading mongoose but the tests are still passing
            return yield (0, exports.systemLogCollection)(conn).distinct('payload.type', { $and: filter });
        });
    },
    create(systemLog) {
        return __awaiter(this, void 0, void 0, function* () {
            const newRow = Object.assign(Object.assign({}, systemLog), { payload: (0, escape_1.escapeObject)(systemLog.payload) });
            yield (0, exports.systemLogCollection)(conn).replaceOne({ id: newRow.id }, newRow, { session: conn.session, upsert: true });
        });
    },
});
exports.createSystemLogModel = createSystemLogModel;
//# sourceMappingURL=systemLogModel.js.map