"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.createBulkOperationModel = exports.prepareBulkOperationTable = exports.bulkOperationCollection = void 0;
const bulkOperation_utils_1 = require("../../Schema/BulkOperation/bulkOperation.utils");
const Enums_1 = require("@signageos/common-types/dist/BulkOperation/Enums");
const utils_1 = require("../Lib/utils");
const collections_1 = require("../Lib/collections");
const lodash_1 = require("lodash");
const bulkOperationCollection = (conn) => conn.connection.collection(collections_1.Collection.BulkOperation);
exports.bulkOperationCollection = bulkOperationCollection;
const prepareBulkOperationTable = (conn) => __awaiter(void 0, void 0, void 0, function* () {
    yield (0, exports.bulkOperationCollection)(conn).createIndex({ uid: 1 }, { name: 'uid', unique: true, background: true });
    yield (0, exports.bulkOperationCollection)(conn).createIndex({ organizationUid: 1 }, {
        name: 'organizationUid',
        background: true,
    });
    yield (0, exports.bulkOperationCollection)(conn).createIndex({ name: 1 }, { name: 'name', background: true });
    yield (0, exports.bulkOperationCollection)(conn).createIndex({ organizationUid: 1, isDraft: 1, name: 'text' }, {
        name: 'organizationUid_isDraft_name',
        background: true,
    });
    yield (0, exports.bulkOperationCollection)(conn).createIndex({ finishedAt: 1 }, { name: 'finishedAt', background: true });
});
exports.prepareBulkOperationTable = prepareBulkOperationTable;
const validateBulkOperationIsDraft = (conn, bulkOperationUid) => __awaiter(void 0, void 0, void 0, function* () {
    const bulkOperation = yield (0, exports.bulkOperationCollection)(conn).findOne({ uid: bulkOperationUid }, { session: conn.session });
    if (!bulkOperation) {
        throw new Error(`Bulk operation with uid ${bulkOperationUid} not found`);
    }
    if (!(bulkOperation === null || bulkOperation === void 0 ? void 0 : bulkOperation.isDraft)) {
        throw new Error(`Bulk operation with uid ${bulkOperationUid} cannot be updated. It's not a draft`);
    }
});
const createFilterQueryFromPropertyFilter = ({ organizationUids, fetchDrafts, filter, }) => {
    const filterQuery = {};
    filterQuery.organizationUids = { $in: organizationUids };
    filterQuery.isDraft = fetchDrafts ? { $eq: true } : { $ne: true };
    if (filter === null || filter === void 0 ? void 0 : filter.name) {
        filterQuery.name = { $regex: (0, lodash_1.escapeRegExp)(filter.name), $options: 'i' };
    }
    if (filter === null || filter === void 0 ? void 0 : filter.currentStatus) {
        filterQuery.currentStatus = { $in: filter.currentStatus };
    }
    return filterQuery;
};
const createBulkOperationModel = (conn) => {
    return {
        fetchByUid(bulkOperationId) {
            return __awaiter(this, void 0, void 0, function* () {
                return (0, exports.bulkOperationCollection)(conn).findOne({ uid: bulkOperationId }, { session: conn.session });
            });
        },
        fetchByOrganizationUids(organizationUids, paginationAndSorting, fetchDrafts = false, filter) {
            return __awaiter(this, void 0, void 0, function* () {
                const matcher = createFilterQueryFromPropertyFilter({ organizationUids, fetchDrafts, filter });
                const sorting = (0, bulkOperation_utils_1.getSorting)(paginationAndSorting === null || paginationAndSorting === void 0 ? void 0 : paginationAndSorting.sorting);
                const pipeline = (0, utils_1.getPipeline)({ matcher, sorting, pagination: paginationAndSorting === null || paginationAndSorting === void 0 ? void 0 : paginationAndSorting.pagination });
                return (0, exports.bulkOperationCollection)(conn).aggregate(pipeline, { session: conn.session }).toArray();
            });
        },
        fetchCountByOrganizationUids(organizationUids, fetchDrafts = false, filter) {
            var _a;
            return __awaiter(this, void 0, void 0, function* () {
                const matcher = createFilterQueryFromPropertyFilter({ organizationUids, fetchDrafts, filter });
                const pipeline = [
                    {
                        $match: matcher,
                    },
                    { $count: 'count' },
                ];
                const aggregationResultCursor = (0, exports.bulkOperationCollection)(conn).aggregate(pipeline, {
                    session: conn.session,
                });
                if (yield aggregationResultCursor.hasNext()) {
                    const row = yield aggregationResultCursor.next();
                    return (_a = row === null || row === void 0 ? void 0 : row.count) !== null && _a !== void 0 ? _a : 0;
                }
                return 0;
            });
        },
        create(record) {
            return __awaiter(this, void 0, void 0, function* () {
                yield (0, exports.bulkOperationCollection)(conn).insertOne(record, { session: conn.session });
            });
        },
        updateFailedDevice(bulkOperationId, deviceUid, decrementInProgress = true) {
            return __awaiter(this, void 0, void 0, function* () {
                const decrementValue = decrementInProgress ? -1 : 0;
                const updateExpr = {
                    $inc: {
                        'progress.failed': 1,
                        'progress.inProgress': decrementValue,
                    },
                    $addToSet: { failedDeviceUids: deviceUid },
                    $pull: {
                        inProgressDeviceUids: deviceUid,
                    },
                };
                yield (0, exports.bulkOperationCollection)(conn).updateOne({
                    uid: bulkOperationId,
                    failedDeviceUids: {
                        $ne: deviceUid,
                    },
                }, updateExpr, {
                    session: conn.session,
                });
            });
        },
        updateSuccessfulDevice(bulkOperationId, deviceUid, decrementInProgress = true) {
            return __awaiter(this, void 0, void 0, function* () {
                const decrementValue = decrementInProgress ? -1 : 0;
                const updateExpr = {
                    $inc: {
                        'progress.succeeded': 1,
                        'progress.inProgress': decrementValue,
                    },
                    $addToSet: { successfulDeviceUids: deviceUid },
                    $pull: {
                        inProgressDeviceUids: deviceUid,
                    },
                };
                yield (0, exports.bulkOperationCollection)(conn).updateOne({
                    uid: bulkOperationId,
                    successfulDeviceUids: { $ne: deviceUid },
                }, updateExpr, {
                    session: conn.session,
                });
            });
        },
        updateSkippedDevice(bulkOperationId, deviceUid, decrementInProgress = true) {
            return __awaiter(this, void 0, void 0, function* () {
                const decrementValue = decrementInProgress ? -1 : 0;
                const updateExpr = {
                    $inc: {
                        'progress.skipped': 1,
                        'progress.inProgress': decrementValue,
                    },
                    $addToSet: { skippedDeviceUids: deviceUid },
                    $pull: {
                        inProgressDeviceUids: deviceUid,
                    },
                };
                yield (0, exports.bulkOperationCollection)(conn).updateOne({
                    uid: bulkOperationId,
                    skippedDeviceUids: { $ne: deviceUid },
                }, updateExpr, {
                    session: conn.session,
                });
            });
        },
        updateInProgressDevice(bulkOperationId, deviceUid, updateValue) {
            return __awaiter(this, void 0, void 0, function* () {
                if (updateValue > 0) {
                    yield (0, exports.bulkOperationCollection)(conn).updateOne({
                        uid: bulkOperationId,
                        inProgressDeviceUids: { $ne: deviceUid },
                    }, {
                        $inc: {
                            'progress.inProgress': updateValue,
                        },
                        $addToSet: {
                            inProgressDeviceUids: deviceUid,
                        },
                    }, {
                        session: conn.session,
                    });
                }
                else {
                    yield (0, exports.bulkOperationCollection)(conn).updateOne({
                        uid: bulkOperationId,
                        inProgressDeviceUids: { $exists: !!deviceUid },
                    }, {
                        $inc: {
                            'progress.inProgress': updateValue,
                        },
                        $pull: {
                            inProgressDeviceUids: deviceUid,
                        },
                    }, {
                        session: conn.session,
                    });
                }
            });
        },
        updateDateRecord(bulkOperationId, updateObject) {
            return __awaiter(this, void 0, void 0, function* () {
                const updateExpr = {
                    $set: Object.assign({ isRunning: false }, updateObject),
                };
                yield (0, exports.bulkOperationCollection)(conn).updateOne({ uid: bulkOperationId }, updateExpr, {
                    session: conn.session,
                });
            });
        },
        updateIsRunning(bulkOperationId, updateValue) {
            return __awaiter(this, void 0, void 0, function* () {
                const updateExpr = {
                    $set: {
                        isRunning: updateValue,
                    },
                };
                yield (0, exports.bulkOperationCollection)(conn).updateOne({ uid: bulkOperationId }, updateExpr, {
                    session: conn.session,
                });
            });
        },
        updateRollingUpdate(bulkOperationId, update) {
            return __awaiter(this, void 0, void 0, function* () {
                const updateExpr = {
                    $set: {
                        rollingUpdate: update,
                    },
                };
                yield (0, exports.bulkOperationCollection)(conn).updateOne({ uid: bulkOperationId }, updateExpr, {
                    session: conn.session,
                });
            });
        },
        setCurrentStatusToWaiting(bulkOperationId) {
            return __awaiter(this, void 0, void 0, function* () {
                const updateExpr = {
                    $set: {
                        currentStatus: Enums_1.BulkOperationStatus.WAITING,
                    },
                };
                yield (0, exports.bulkOperationCollection)(conn).updateOne({ uid: bulkOperationId }, updateExpr, {
                    session: conn.session,
                });
            });
        },
        fetchActiveBulkOperations(date) {
            return __awaiter(this, void 0, void 0, function* () {
                return (0, exports.bulkOperationCollection)(conn)
                    .aggregate([
                    {
                        $match: {
                            finishedAt: null,
                            stoppedAt: null,
                            archivedAt: null,
                            isRunning: false,
                            isDraft: { $ne: true },
                            $and: [
                                {
                                    $or: [
                                        {
                                            'schedule.datetime': { $lt: date },
                                        },
                                        {
                                            schedule: null,
                                        },
                                    ],
                                },
                                {
                                    $or: [
                                        { pausedAt: null },
                                        {
                                            $and: [
                                                { resumedAt: { $exists: true } },
                                                {
                                                    $expr: { $gt: ['$resumedAt', '$pausedAt'] },
                                                },
                                            ],
                                        },
                                    ],
                                },
                            ],
                        },
                    },
                ], { session: conn.session })
                    .toArray();
            });
        },
        addDeviceToDraft(bulkOperationUid, deviceIdentityHashes) {
            return __awaiter(this, void 0, void 0, function* () {
                yield validateBulkOperationIsDraft(conn, bulkOperationUid);
                yield (0, exports.bulkOperationCollection)(conn).updateOne({ uid: bulkOperationUid }, { $addToSet: { 'filter.identityHashes': { $each: deviceIdentityHashes } } }, { session: conn.session });
            });
        },
        finishDraft(bulkOperationUid) {
            return __awaiter(this, void 0, void 0, function* () {
                yield validateBulkOperationIsDraft(conn, bulkOperationUid);
                yield (0, exports.bulkOperationCollection)(conn).updateOne({ uid: bulkOperationUid }, { $unset: { isDraft: 1 } }, { session: conn.session });
            });
        },
        addDeviceUids(bulkOperationUid, deviceUids) {
            return __awaiter(this, void 0, void 0, function* () {
                yield (0, exports.bulkOperationCollection)(conn).updateOne({ uid: bulkOperationUid }, { $addToSet: { deviceUids: { $each: deviceUids } } }, { session: conn.session });
            });
        },
    };
};
exports.createBulkOperationModel = createBulkOperationModel;
//# sourceMappingURL=bulkOperationModel.js.map