"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.createAppletModel = exports.prepareAppletTable = exports.appletCollection = void 0;
const collections_1 = require("../Lib/collections");
const organization_utils_1 = require("../Organization/organization.utils");
const appletVersionModel_1 = require("./appletVersionModel");
const deviceModel_utils_1 = require("../Device/deviceModel.utils");
const deviceModel_1 = require("../Device/deviceModel");
const lodash_1 = require("lodash");
const appletCollection = (conn) => conn.connection.collection(collections_1.Collection.Applet);
exports.appletCollection = appletCollection;
const prepareAppletTable = (conn) => __awaiter(void 0, void 0, void 0, function* () {
    yield (0, exports.appletCollection)(conn).createIndex('uid', { name: 'uid', unique: true, background: true });
    yield (0, exports.appletCollection)(conn).createIndex('organizationUid', { name: 'organizationUid', background: true });
    yield (0, exports.appletCollection)(conn).createIndex('versions.id', { name: 'versions.id', background: true });
    yield (0, exports.appletCollection)(conn).createIndex('versions.version', { name: 'versions.version', background: true });
});
exports.prepareAppletTable = prepareAppletTable;
const createAppletModel = (conn, ...[deviceInfoModel]) => ({
    create(uid, name, createdAt, organizationUid, iconUrl) {
        return __awaiter(this, void 0, void 0, function* () {
            const doc = { uid, name, createdAt, organizationUid, iconUrl };
            yield (0, exports.appletCollection)(conn).insertOne(doc, { session: conn.session });
        });
    },
    fetchByUid(uid) {
        return __awaiter(this, void 0, void 0, function* () {
            return (0, exports.appletCollection)(conn).findOne({ uid: uid }, { session: conn.session });
        });
    },
    updateName(appletDoc, name) {
        return __awaiter(this, void 0, void 0, function* () {
            yield (0, exports.appletCollection)(conn).updateOne({ uid: appletDoc.uid }, { $set: { name } }, { session: conn.session });
        });
    },
    updateIconUrl(appletDoc, iconUrl) {
        return __awaiter(this, void 0, void 0, function* () {
            yield (0, exports.appletCollection)(conn).updateOne({ uid: appletDoc.uid }, { $set: { iconUrl } }, { session: conn.session });
        });
    },
    remove(applet) {
        return __awaiter(this, void 0, void 0, function* () {
            const versionModel = (0, appletVersionModel_1.createAppletVersionModel)(conn);
            const versions = yield versionModel.fetchListByApplet(applet);
            if (versions.length > 0) {
                throw new Error(`Cannot remove Applet. ${versions.length} applet versions exist.`);
            }
            yield (0, exports.appletCollection)(conn).deleteOne({ uid: applet.uid }, { session: conn.session });
        });
    },
    updateOrganization(appletDoc, organization) {
        return __awaiter(this, void 0, void 0, function* () {
            yield (0, exports.appletCollection)(conn).updateOne({ uid: appletDoc.uid }, { $set: { organizationUid: organization.uid } }, { session: conn.session });
        });
    },
    fetchListByOrganization(organization) {
        return __awaiter(this, void 0, void 0, function* () {
            return (0, exports.appletCollection)(conn).find({ organizationUid: organization.uid }, { session: conn.session }).toArray();
        });
    },
    fetchListByOrganizations(organizations) {
        return __awaiter(this, void 0, void 0, function* () {
            return (0, exports.appletCollection)(conn)
                .find({
                organizationUid: {
                    $in: organizations.map((organization) => organization.uid),
                },
            }, { session: conn.session })
                .toArray();
        });
    },
    fetchSortedListByOrganizations(organizations, options) {
        return __awaiter(this, void 0, void 0, function* () {
            const { sorter, filter, pagination } = options;
            const pipeline = [];
            pipeline.push({
                $match: {
                    organizationUid: {
                        $in: organizations.map((organization) => organization.uid),
                    },
                },
            });
            if (filter && filter.name) {
                const names = filter.name.split(/\s+/);
                pipeline.push({
                    $match: {
                        $and: names.map((name) => {
                            return { name: { $regex: (0, lodash_1.escapeRegExp)(name), $options: 'i' } };
                        }),
                    },
                });
            }
            if (filter === null || filter === void 0 ? void 0 : filter.uids) {
                pipeline.push({
                    $match: {
                        uid: { $in: filter.uids },
                    },
                });
            }
            if (sorter) {
                if (sorter.organizationTitle) {
                    pipeline.push({
                        $lookup: {
                            from: 'organization',
                            localField: 'organizationUid',
                            foreignField: 'uid',
                            as: 'organization',
                        },
                    });
                }
                const getSorter = () => {
                    if (sorter.name) {
                        return { name: sorter.name };
                    }
                    else if (sorter.createdAt) {
                        return { createdAt: sorter.createdAt };
                    }
                    else if (sorter.organizationTitle) {
                        return { 'organization.title': sorter.organizationTitle };
                    }
                    return { createdAt: -1 };
                };
                pipeline.push({ $sort: getSorter() });
            }
            else {
                pipeline.push({ $sort: { createdAt: -1 } });
            }
            if (pagination === null || pagination === void 0 ? void 0 : pagination.offset) {
                pipeline.push({ $skip: pagination.offset });
            }
            if (pagination === null || pagination === void 0 ? void 0 : pagination.limit) {
                pipeline.push({ $limit: pagination.limit });
            }
            pipeline.push({ $project: { organization: 0 } });
            return (0, exports.appletCollection)(conn).aggregate(pipeline, { session: conn.session }).toArray();
        });
    },
    fetchTotalCountByOrganizations(organizations, filter) {
        var _a;
        return __awaiter(this, void 0, void 0, function* () {
            let pipeline = [];
            pipeline.push({
                $match: {
                    organizationUid: {
                        $in: organizations.map((organization) => organization.uid),
                    },
                },
            });
            if (filter && filter.name) {
                const names = filter.name.split(/\s+/);
                pipeline.push({
                    $match: {
                        $and: names.map((name) => {
                            return { name: { $regex: (0, lodash_1.escapeRegExp)(name), $options: 'i' } };
                        }),
                    },
                });
            }
            if (filter === null || filter === void 0 ? void 0 : filter.uids) {
                pipeline.push({
                    $match: {
                        uid: { $in: filter.uids },
                    },
                });
            }
            pipeline.push({ $count: 'count' });
            const [result] = yield (0, exports.appletCollection)(conn).aggregate(pipeline, { session: conn.session }).toArray();
            return (_a = result === null || result === void 0 ? void 0 : result.count) !== null && _a !== void 0 ? _a : 0;
        });
    },
    fetchListByDeviceFilter(organizations, deviceFilter = {}) {
        return __awaiter(this, void 0, void 0, function* () {
            if (organizations.length === 0) {
                return [];
            }
            const organizationFilter = (0, organization_utils_1.getOrganizationFilter)(organizations);
            const filter = yield (0, deviceModel_utils_1.convertPropertyFilterToMongoFilter)({
                sourceMongoFilter: organizationFilter,
                propertyFilter: deviceFilter,
                deviceInfoModel,
            });
            const { pipeline } = (0, deviceModel_utils_1.getFetchListByOrganizationsProjection)({ filter: deviceFilter });
            return (0, deviceModel_1.deviceCollection)(conn)
                .aggregate([
                ...pipeline,
                { $match: filter },
                {
                    $project: { identityHash: 1 },
                },
                {
                    $lookup: {
                        from: 'timing',
                        localField: 'identityHash',
                        foreignField: 'deviceIdentityHash',
                        as: 'timing',
                    },
                },
                {
                    $match: {
                        timing: { $exists: true, $not: { $size: 0 } },
                    },
                },
                {
                    $project: { timing: 1 },
                },
                {
                    $unwind: '$timing',
                },
                {
                    $project: { appletUid: '$timing.appletUid' },
                },
                {
                    $group: { _id: '$appletUid' },
                },
                {
                    $lookup: {
                        from: 'applet',
                        localField: '_id',
                        foreignField: 'uid',
                        as: 'applet',
                    },
                },
                {
                    $unwind: '$applet',
                },
                { $replaceRoot: { newRoot: '$applet' } },
                {
                    $project: { _id: 0 },
                },
                { $sort: { name: 1 } },
            ], { session: conn.session })
                .toArray();
        });
    },
    fetchCommonListByDeviceFilter(organizations, deviceFilter = {}) {
        return __awaiter(this, void 0, void 0, function* () {
            if (organizations.length === 0) {
                return [];
            }
            const organizationFilter = (0, organization_utils_1.getOrganizationFilter)(organizations);
            const filter = yield (0, deviceModel_utils_1.convertPropertyFilterToMongoFilter)({
                sourceMongoFilter: organizationFilter,
                propertyFilter: deviceFilter,
                deviceInfoModel,
            });
            const { pipeline } = (0, deviceModel_utils_1.getFetchListByOrganizationsProjection)({ filter: deviceFilter });
            return (0, deviceModel_1.deviceCollection)(conn)
                .aggregate([
                ...pipeline,
                { $match: filter },
                {
                    $project: { identityHash: 1 },
                },
                {
                    $lookup: {
                        from: 'timing',
                        localField: 'identityHash',
                        foreignField: 'deviceIdentityHash',
                        as: 'deviceTiming',
                    },
                },
                {
                    $facet: {
                        // get device count per applet
                        appletsDeviceCount: [
                            {
                                $unwind: '$deviceTiming',
                            },
                            {
                                $group: { _id: '$deviceTiming.appletUid', count: { $sum: 1 } },
                            },
                        ],
                        // get total device count
                        totalDeviceCount: [{ $count: 'count' }],
                    },
                },
                {
                    $project: {
                        appletsUids: {
                            $map: {
                                input: {
                                    $filter: {
                                        input: '$appletsDeviceCount',
                                        as: 'appletDeviceCountFilter',
                                        // if there is an applet with the same device count as the total device count, we have found a common applet for all devices
                                        cond: { $eq: ['$$appletDeviceCountFilter.count', { $arrayElemAt: ['$totalDeviceCount.count', 0] }] },
                                    },
                                },
                                as: 'appletDeviceCount',
                                in: '$$appletDeviceCount._id',
                            },
                        },
                    },
                },
                {
                    $match: { $expr: { $gt: [{ $size: '$appletsUids' }, 0] } },
                },
                {
                    $lookup: {
                        from: 'applet',
                        localField: 'appletsUids',
                        foreignField: 'uid',
                        as: 'applets',
                    },
                },
                {
                    $unwind: '$applets',
                },
                {
                    $replaceRoot: { newRoot: '$applets' },
                },
                {
                    $project: { _id: 0 },
                },
                { $sort: { name: 1 } },
            ], { session: conn.session })
                .toArray();
        });
    },
});
exports.createAppletModel = createAppletModel;
//# sourceMappingURL=appletModel.js.map