"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.createPipelineFromAccountPropertyFilter = exports.createAccountModel = exports.prepareAccountTable = void 0;
const Account_1 = require("@signageos/common-types/dist/Account/Account");
const accountModel_1 = require("../../Schema/Account/accountModel");
const organizationModel_1 = require("../Organization/organizationModel");
const collections_1 = require("../Lib/collections");
const paginationTypes_1 = require("../../Lib/Pagination/paginationTypes");
const pagination_1 = require("../../Lib/Pagination/pagination");
const utils_1 = require("../Lib/utils");
const lodash_1 = require("lodash");
const accountCollection = (conn) => conn.connection.collection(collections_1.Collection.Account);
const prepareAccountTable = (conn) => __awaiter(void 0, void 0, void 0, function* () {
    yield accountCollection(conn).createIndex('id', { name: 'id', unique: true, background: true });
    yield accountCollection(conn).createIndex('username', { name: 'username', background: true });
    yield accountCollection(conn).createIndex('email', { name: 'email', unique: true, background: true });
    yield accountCollection(conn).createIndex('sessions.sessionUid', { name: 'sessions.sessionUid', background: true });
    yield accountCollection(conn).createIndex({ 'privileges.entity': 1, 'privileges.entityUid': 1 }, { name: 'privileges.entity_privileges.entityUid', background: true });
});
exports.prepareAccountTable = prepareAccountTable;
const createAccountModel = (conn) => {
    /** ensureAccount is responsible that IAccount is semantically valid.
     *   - Checks mandatory fields exist
     *   - Fills default values to fields that are inconsistent in the storage
     */
    const ensureAccount = (a) => {
        if (a === null) {
            return null;
        }
        if (typeof a.id !== 'number') {
            throw new Error("Invalid Account document. Field 'id' is missing.");
        }
        if (!a.email) {
            throw new Error("Invalid Account document. Field 'email' is missing.");
        }
        if (!a.accessLevel) {
            throw new Error("Invalid Account document. Field 'accessLevel' is missing.");
        }
        if (typeof a.privileges === 'undefined') {
            a.privileges = [];
        }
        return a;
    };
    return {
        create(params) {
            return __awaiter(this, void 0, void 0, function* () {
                const newDoc = Object.assign(Object.assign({}, params), { sessions: [], privileges: [], isActive: false });
                yield accountCollection(conn).insertOne(newDoc, { session: conn.session });
            });
        },
        fetchById(id) {
            return __awaiter(this, void 0, void 0, function* () {
                return ensureAccount(yield accountCollection(conn).findOne({ id }, { session: conn.session }));
            });
        },
        fetchBySessionUid(sessionUid) {
            return __awaiter(this, void 0, void 0, function* () {
                return ensureAccount(yield accountCollection(conn).findOne({ 'sessions.sessionUid': { $in: [sessionUid] } }, { session: conn.session }));
            });
        },
        fetchByUsername(username, skipArchived) {
            return __awaiter(this, void 0, void 0, function* () {
                return ensureAccount(yield accountCollection(conn).findOne(Object.assign({ username }, (skipArchived ? { accessLevel: { $ne: Account_1.AccessLevel.Archived } } : {})), { session: conn.session }));
            });
        },
        fetchByEmail(email, skipArchived) {
            return __awaiter(this, void 0, void 0, function* () {
                return ensureAccount(yield accountCollection(conn).findOne(Object.assign({ email: { $regex: `^${(0, lodash_1.escapeRegExp)(email)}$`, $options: 'i' } }, (skipArchived ? { accessLevel: { $ne: Account_1.AccessLevel.Archived } } : {})), { session: conn.session }));
            });
        },
        fetchByEmailCaseSensitive(email) {
            return __awaiter(this, void 0, void 0, function* () {
                return ensureAccount(yield accountCollection(conn).findOne({ email: { $regex: `^${(0, lodash_1.escapeRegExp)(email)}$` } }, { session: conn.session }));
            });
        },
        fetchListByIds(ids) {
            return __awaiter(this, void 0, void 0, function* () {
                const accounts = yield accountCollection(conn)
                    .find({ id: { $in: ids } }, { session: conn.session })
                    .toArray();
                return accounts.map(ensureAccount);
            });
        },
        fetchFilteredList(filter, pagination) {
            return __awaiter(this, void 0, void 0, function* () {
                const pipeline = createPipelineFromAccountPropertyFilter(filter);
                pipeline.push({
                    $sort: {
                        createdAt: 1,
                    },
                });
                if (pagination === null || pagination === void 0 ? void 0 : pagination.offset) {
                    pipeline.push({ $skip: pagination.offset });
                }
                if (pagination === null || pagination === void 0 ? void 0 : pagination.limit) {
                    pipeline.push({ $limit: pagination.limit });
                }
                return accountCollection(conn).aggregate(pipeline, { session: conn.session }).toArray();
            });
        },
        fetchAllList() {
            return __awaiter(this, void 0, void 0, function* () {
                const accounts = yield accountCollection(conn).find({}, { session: conn.session }).sort({ username: 1 }).toArray();
                return accounts.map(ensureAccount);
            });
        },
        fetchSortedList(options) {
            return __awaiter(this, void 0, void 0, function* () {
                const { sorter, filter, pagination } = options;
                const pipeline = createPipelineFromAccountPropertyFilter(filter);
                if (sorter) {
                    const getSorter = () => {
                        if (sorter.name) {
                            return { lastname: sorter.name };
                        }
                        if (sorter.created) {
                            return { createdAt: sorter.created };
                        }
                        return sorter;
                    };
                    pipeline.push({ $sort: getSorter() });
                }
                if (pagination === null || pagination === void 0 ? void 0 : pagination.offset) {
                    pipeline.push({ $skip: pagination.offset });
                }
                if (pagination === null || pagination === void 0 ? void 0 : pagination.limit) {
                    pipeline.push({ $limit: pagination.limit });
                }
                return accountCollection(conn).aggregate(pipeline, { session: conn.session }).toArray();
            });
        },
        fetchTotalCount(filter) {
            var _a;
            return __awaiter(this, void 0, void 0, function* () {
                const pipeline = createPipelineFromAccountPropertyFilter(filter);
                pipeline.push({ $count: 'count' });
                const [result] = yield accountCollection(conn).aggregate(pipeline, { session: conn.session }).toArray();
                return (_a = result === null || result === void 0 ? void 0 : result.count) !== null && _a !== void 0 ? _a : 0;
            });
        },
        /**
         * @param companyUid uid of organization entity which is company
         * @returns array of promise accounts, could be empty if nothing was found
         */
        fetchAllCompanyAccounts(companyUid) {
            return __awaiter(this, void 0, void 0, function* () {
                const companyAccounts = accountCollection(conn)
                    .aggregate([
                    {
                        $lookup: {
                            from: 'organization',
                            localField: 'id',
                            foreignField: 'accountIds',
                            as: 'orgs',
                        },
                    },
                    {
                        $match: {
                            'orgs.uid': {
                                $in: [companyUid, ...(yield (0, organizationModel_1.organizationCollection)(conn).distinct('uid', { parentUid: companyUid }))],
                            },
                        },
                    },
                    {
                        $project: {
                            orgs: 0,
                        },
                    },
                ], { session: conn.session })
                    .toArray();
                return companyAccounts;
            });
        },
        fetchPrivilegesByOrganizationUid(organizationUid, accountId, paginationAndSorting) {
            return __awaiter(this, void 0, void 0, function* () {
                let pipeline = [];
                if (accountId) {
                    pipeline.push({ $match: { id: accountId } });
                }
                pipeline = [
                    ...pipeline,
                    { $match: { 'privileges.entityUid': organizationUid } },
                    {
                        $project: {
                            _id: 0,
                            account: '$$ROOT',
                            privilege: {
                                $filter: {
                                    input: '$privileges',
                                    as: 'privilege',
                                    cond: { $eq: ['$$privilege.entityUid', organizationUid] },
                                },
                            },
                        },
                    },
                    { $unwind: '$privilege' },
                ];
                const sorting = (0, pagination_1.getSorting)({
                    defaultSorting: {
                        field: 'privilege.assignedAt',
                        order: paginationTypes_1.SortOrder.DESC,
                    },
                    sorting: paginationAndSorting === null || paginationAndSorting === void 0 ? void 0 : paginationAndSorting.sorting,
                });
                const sortPipeline = (0, utils_1.getPipeline)({ sorting, pagination: paginationAndSorting === null || paginationAndSorting === void 0 ? void 0 : paginationAndSorting.pagination });
                const result = yield accountCollection(conn).aggregate([...pipeline, ...sortPipeline], {
                    session: conn.session,
                });
                return result.toArray();
            });
        },
        updateEmail(accountRow, email) {
            return __awaiter(this, void 0, void 0, function* () {
                yield accountCollection(conn).updateOne({ id: accountRow.id }, {
                    $set: {
                        email,
                    },
                }, { session: conn.session });
            });
        },
        updatePasswordAndSalt(accountRow, password, salt) {
            return __awaiter(this, void 0, void 0, function* () {
                yield accountCollection(conn).updateOne({ id: accountRow.id }, {
                    $set: {
                        password,
                        salt,
                    },
                }, { session: conn.session });
            });
        },
        updateFirstNameAndLastName(accountRow, firstname, lastname) {
            return __awaiter(this, void 0, void 0, function* () {
                yield accountCollection(conn).updateOne({ id: accountRow.id }, {
                    $set: {
                        firstname,
                        lastname,
                    },
                }, { session: conn.session });
            });
        },
        updateAccessLevel(accountRow, accessLevel) {
            return __awaiter(this, void 0, void 0, function* () {
                yield accountCollection(conn).updateOne({ id: accountRow.id }, {
                    $set: {
                        accessLevel,
                    },
                }, { session: conn.session });
            });
        },
        addSession(accountRow, sessionUid, createdAt) {
            return __awaiter(this, void 0, void 0, function* () {
                yield accountCollection(conn).updateOne({ id: accountRow.id }, {
                    $addToSet: { sessions: { sessionUid, createdAt } },
                }, { session: conn.session });
            });
        },
        removeSession(accountRow, sessionUid) {
            return __awaiter(this, void 0, void 0, function* () {
                yield accountCollection(conn).updateOne({ id: accountRow.id }, {
                    $pull: { sessions: { sessionUid } },
                }, { session: conn.session });
            });
        },
        removeAllSessions(accountRow) {
            return __awaiter(this, void 0, void 0, function* () {
                yield accountCollection(conn).updateOne({ id: accountRow.id }, {
                    $pull: { sessions: {} },
                }, { session: conn.session });
            });
        },
        setActive(accountRow, active) {
            return __awaiter(this, void 0, void 0, function* () {
                yield accountCollection(conn).updateOne({ id: accountRow.id }, {
                    $set: {
                        isActive: active,
                    },
                }, { session: conn.session });
            });
        },
        addPrivilege(accountRow, privilege) {
            return __awaiter(this, void 0, void 0, function* () {
                // Note: unfortunately the $pull and $addToSet commands cannot be run in single query
                //       https://stackoverflow.com/questions/24300148/pull-and-addtoset-at-the-same-time-with-mongo
                //       update: {
                // 		     $pull: { privileges: { entity, entityUid } },
                //           $addToSet: { privileges: newPrivilegeDoc },
                // 		 }
                yield accountCollection(conn).updateOne({ id: accountRow.id }, { $pull: { privileges: { entity: privilege.entity, entityUid: privilege.entityUid } } }, { session: conn.session });
                yield accountCollection(conn).updateOne({ id: accountRow.id }, { $addToSet: { privileges: privilege } }, { session: conn.session });
            });
        },
        removePrivilege(accountRow, privilege) {
            return __awaiter(this, void 0, void 0, function* () {
                yield accountCollection(conn).updateOne({ id: accountRow.id }, {
                    $pull: { privileges: { entity: privilege.entity, entityUid: privilege.entityUid } },
                }, { session: conn.session });
            });
        },
    };
};
exports.createAccountModel = createAccountModel;
function createPipelineFromAccountPropertyFilter(propertyFilter) {
    var _a;
    const pipeline = [];
    if (propertyFilter === null || propertyFilter === void 0 ? void 0 : propertyFilter.fulltext) {
        const values = propertyFilter.fulltext.split(/\s+/);
        pipeline.push({
            $match: {
                $and: values.map((fulltext) => {
                    return {
                        $or: [
                            { email: { $regex: (0, lodash_1.escapeRegExp)(fulltext), $options: 'i' } },
                            { firstname: { $regex: (0, lodash_1.escapeRegExp)(fulltext), $options: 'i' } },
                            { lastname: { $regex: (0, lodash_1.escapeRegExp)(fulltext), $options: 'i' } },
                            { id: parseInt(fulltext, 10) },
                        ],
                    };
                }),
            },
        });
    }
    if (propertyFilter === null || propertyFilter === void 0 ? void 0 : propertyFilter.type) {
        pipeline.push({
            $match: { accessLevel: { $eq: propertyFilter.type } },
        });
    }
    const organizationFilterConditions = [];
    if ((_a = propertyFilter === null || propertyFilter === void 0 ? void 0 : propertyFilter.organizationUids) === null || _a === void 0 ? void 0 : _a.length) {
        organizationFilterConditions.push({
            privileges: {
                $elemMatch: {
                    entity: accountModel_1.PrivilegeEntity.Organization,
                    entityUid: { $in: propertyFilter.organizationUids },
                },
            },
        });
    }
    if ((propertyFilter === null || propertyFilter === void 0 ? void 0 : propertyFilter.accountsWithoutOrganization) !== undefined) {
        if (propertyFilter.accountsWithoutOrganization) {
            organizationFilterConditions.push({
                'privileges.entity': {
                    $ne: accountModel_1.PrivilegeEntity.Organization,
                },
            });
        }
        else {
            organizationFilterConditions.push({
                'privileges.entity': accountModel_1.PrivilegeEntity.Organization,
            });
        }
    }
    if (organizationFilterConditions.length) {
        pipeline.push({
            $match: {
                $or: organizationFilterConditions,
            },
        });
    }
    if (propertyFilter === null || propertyFilter === void 0 ? void 0 : propertyFilter.rolesInCompany) {
        pipeline.push({
            $match: {
                privileges: {
                    $elemMatch: {
                        role: { $in: propertyFilter.rolesInCompany.roles },
                        entity: accountModel_1.PrivilegeEntity.Company,
                        entityUid: propertyFilter.rolesInCompany.companyUid,
                    },
                },
            },
        });
    }
    if (propertyFilter === null || propertyFilter === void 0 ? void 0 : propertyFilter.emailDomains) {
        pipeline.push({
            $match: {
                email: {
                    $regex: new RegExp(propertyFilter.emailDomains.map((emailDomain) => `@${(0, lodash_1.escapeRegExp)(emailDomain)}$`).join('|')),
                    $options: 'i',
                },
            },
        });
    }
    if (propertyFilter === null || propertyFilter === void 0 ? void 0 : propertyFilter.ids) {
        pipeline.push({ $match: { id: { $in: propertyFilter === null || propertyFilter === void 0 ? void 0 : propertyFilter.ids } } });
    }
    if (propertyFilter === null || propertyFilter === void 0 ? void 0 : propertyFilter.excludeAdmins) {
        pipeline.push({
            $match: { accessLevel: { $ne: Account_1.AccessLevel.Admin } },
        });
    }
    return pipeline;
}
exports.createPipelineFromAccountPropertyFilter = createPipelineFromAccountPropertyFilter;
//# sourceMappingURL=accountModel.js.map