"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.monitoringPointCreator = exports.createQueryable = void 0;
const _ = require("lodash");
const influxdb_client_1 = require("@influxdata/influxdb-client");
const versions_1 = require("../../../Helpers/versions");
const expressionCreator_1 = require("../../../Helpers/expressionCreator");
const object_1 = require("@signageos/lib/dist/Utils/object");
const Debug = require("debug");
const dataType_1 = require("../../../Schema/Device/MonitoringLog/dataType");
const errors_1 = require("../../../Schema/Device/MonitoringLog/errors");
const DeviceTelemetryType_1 = require("@signageos/common-types/dist/Device/Telemetry/DeviceTelemetryType");
const debug = Debug('@signageos/user-domain-model:Influx:MonitoringLog:helper');
function createQueryable(type) {
    return new UniversalMonitoringLog(type);
}
exports.createQueryable = createQueryable;
exports.monitoringPointCreator = {
    create(point, type, id, data) {
        var _a;
        point = point.stringField('id', id);
        // TODO back compatibility versionNumber filling (when not available)
        if (type === DeviceTelemetryType_1.DeviceTelemetryType.APPLICATION_VERSION) {
            const appVersionData = data; // nested object infer is not supported
            appVersionData.versionNumber = (_a = appVersionData.versionNumber) !== null && _a !== void 0 ? _a : (0, versions_1.numberizeBigInt)(appVersionData.version);
        }
        const dataTypes = getPropertiesDataTypes(type);
        for (const propertyName in dataTypes) {
            const propertyType = dataTypes[propertyName];
            const value = data === null || data === void 0 ? void 0 : data[propertyName]; // "for in" construction string enumeration is not supported
            const isValid = Object.values(dataType_1.DataType).some((dataType) => {
                if ((0, dataType_1.isOfDataType)(propertyType, dataType)) {
                    switch (dataType) {
                        case dataType_1.DataType.BOOLEAN:
                            return typeof value === 'boolean';
                        case dataType_1.DataType.FLOAT:
                            return typeof value === 'number';
                        case dataType_1.DataType.INT:
                            return typeof value === 'number';
                        case dataType_1.DataType.STRING:
                            return typeof value === 'string';
                        case dataType_1.DataType.TIMESTAMP:
                            return value instanceof Date;
                        case dataType_1.DataType.JSON:
                            return !_.isNil(data) && typeof data === 'object';
                        case dataType_1.DataType.NULL:
                        case dataType_1.DataType.MISSING:
                            return _.isNil(value);
                        default:
                            return false;
                    }
                }
                else {
                    return false;
                }
            });
            if (!isValid) {
                throw new errors_1.InvalidMonitoringLogPropertyValueError(propertyName, propertyType, value !== null && value !== void 0 ? value : data);
            }
            if ((0, dataType_1.isOfDataType)(propertyType, dataType_1.DataType.TIMESTAMP) && value instanceof Date) {
                point = point.uintField(propertyName, value.getTime());
            }
            else if ((0, dataType_1.isOfDataType)(propertyType, dataType_1.DataType.INT) && typeof value === 'number') {
                point = point.intField(propertyName, value);
            }
            else if ((0, dataType_1.isOfDataType)(propertyType, dataType_1.DataType.FLOAT) && typeof value === 'number') {
                point = point.floatField(propertyName, value);
            }
            else if ((0, dataType_1.isOfDataType)(propertyType, dataType_1.DataType.STRING) && typeof value === 'string') {
                point = point.stringField(propertyName, value);
            }
            else if ((0, dataType_1.isOfDataType)(propertyType, dataType_1.DataType.BOOLEAN) && typeof value === 'boolean') {
                point = point.booleanField(propertyName, value);
            }
            else if ((0, dataType_1.isOfDataType)(propertyType, dataType_1.DataType.JSON)) {
                // includes real null & undefined and converts to null
                point = point.stringField(propertyName, JSON.stringify(data !== null && data !== void 0 ? data : null));
            }
            else if ((0, dataType_1.isOfDataType)(propertyType, dataType_1.DataType.NULL) && _.isNil(value)) {
                point = point.stringField(propertyName, '');
            }
            else if ((0, dataType_1.isOfDataType)(propertyType, dataType_1.DataType.MISSING) && _.isNil(value)) {
                // Add no field because it is optional and it may be missing
            }
            else {
                throw new Error(`Unknown data type "${propertyType}" for ${type}, property ${propertyName}.`);
            }
        }
        return point;
    },
};
class UniversalMonitoringLog {
    // eslint-disable-next-line no-empty-function
    constructor(monitoringType) {
        this.monitoringType = monitoringType;
    }
    getLastValueQuery(bucket, deviceIdentityHash) {
        /* eslint-disable max-len */
        const fluxQuery = (0, influxdb_client_1.flux) `
			from(bucket: ${bucket})
			|> range(start: 0)
			|> filter(fn: (r) => r._measurement == ${this.monitoringType} and r.deviceIdentityHash == "${deviceIdentityHash}")
			|> group(columns: ["_field", "_measurement"])
			|> last()
			|> map(fn: (r) => ({ r with organizationUid: if exists r.organizationUid then r.organizationUid else "" }))
			|> pivot(rowKey:["_time", "organizationUid", "deviceIdentityHash", "_measurement"], columnKey: ["_field"], valueColumn: "_value")
			|> rename(columns: { _time: "createdAt", _measurement: "type"})
		`;
        /* eslint-enable max-len */
        return fluxQuery.toString();
    }
    getValuesQuery(bucket, deviceIdentityHash, filter) {
        const start = filter.since ? Math.floor(filter.since.getTime() / 1000) - 1 : 0;
        const stop = filter.until ? Math.floor(filter.until.getTime() / 1000) + 1 : 'now()';
        const fluxQuery = (0, influxdb_client_1.flux) `
			from(bucket: ${bucket})
			|> range(start: ${start}, stop: ${(0, influxdb_client_1.fluxExpression)(stop)})
			|> filter(fn: (r) => r._measurement == ${this.monitoringType} and r.deviceIdentityHash == ${deviceIdentityHash})
			|> sort(columns: ["_time"], desc: true)
			${filter.limit ? (0, influxdb_client_1.flux) `|> limit(n: ${filter.limit})` : (0, influxdb_client_1.fluxExpression)('')}
			|> pivot(rowKey:["_time"], columnKey: ["_field"], valueColumn: "_value")
			|> drop(columns: ["_stop", "_start"])
			|> rename(columns: { _time: "createdAt", _measurement: "type" })
		`;
        return fluxQuery.toString();
    }
    getValuesSinceQuery(bucket, deviceIdentityHash, since) {
        const fluxQuery = (0, influxdb_client_1.flux) `
			from(bucket: ${bucket})
			|> range(start: ${Math.floor(since.getTime() / 1000)})
			|> filter(fn: (r) => r._measurement == ${this.monitoringType} and r.deviceIdentityHash == "${deviceIdentityHash}")
			|> pivot(rowKey:["_time"], columnKey: ["_field"], valueColumn: "_value")
			|> drop(columns: ["_stop", "_start"])
			|> rename(columns: { _time: "createdAt", _measurement: "type" })
		`;
        return fluxQuery.toString();
    }
    getValuesQueryPivoted(bucket, deviceIdentityHash, since, until, limit) {
        const fluxSinceDate = (0, influxdb_client_1.fluxDateTime)(since.toISOString());
        const fluxUntilDate = (0, influxdb_client_1.fluxDateTime)(until.toISOString());
        let fluxQueryStr = (0, influxdb_client_1.flux) `
			from(bucket: ${bucket})
				|> range(start: ${fluxSinceDate}, stop: ${fluxUntilDate})
				|> filter(fn: (r) => r._measurement == ${this.monitoringType} and r.deviceIdentityHash == "${deviceIdentityHash}")
				|> pivot(rowKey:["_time"], columnKey: ["_field"], valueColumn: "_value")
				|> drop(columns: ["_stop", "_start"])
				|> rename(columns: { _time: "createdAt", _measurement: "type" })
		`.toString();
        if (typeof limit === 'number' && limit > 0) {
            fluxQueryStr += `\t\t|> limit(n: ${(0, influxdb_client_1.fluxInteger)(limit)})`;
        }
        return fluxQueryStr;
    }
    getLastNValuesQuery(bucket, deviceIdentityHash, n) {
        const fluxQuery = (0, influxdb_client_1.flux) `
			from(bucket: ${bucket})
			|> range(start: 0)
			|> filter(fn: (r) => r._measurement == ${this.monitoringType} and r.deviceIdentityHash == "${deviceIdentityHash}")
			|> pivot(rowKey:["_time"], columnKey: ["_field"], valueColumn: "_value")
			|> tail(n: ${n})
			|> drop(columns: ["_stop", "_start"])
			|> rename(columns: { _time: "createdAt", _measurement: "type" })
		`;
        return fluxQuery.toString();
    }
    getLastValuesSinceQuery(bucket, deviceIdentityHashes, since) {
        const multipleOrExpression = (0, expressionCreator_1.createEqualsMultipleOrExpression)('r', 'deviceIdentityHash', deviceIdentityHashes);
        const fluxQuery = (0, influxdb_client_1.flux) `
			from(bucket: ${bucket})
				|> range(start: ${(0, influxdb_client_1.fluxDateTime)(since.toISOString())})
				|> filter(fn: (r) => r["_measurement"] == ${this.monitoringType}
					and (${(0, influxdb_client_1.fluxExpression)(multipleOrExpression)})
				)
				|> last()
				|> pivot(rowKey:["_time"], columnKey: ["_field"], valueColumn: "_value")
				|> group()
				|> drop(columns: ["_stop", "_start"])
				|> rename(columns: { _time: "createdAt", _measurement: "type" })
		`;
        return fluxQuery.toString();
    }
    transformPivoted(rows) {
        const monitoringLogsLikes = rows.filter(isMonitoringLog);
        return this.convertMonitoringLogLikesToMonitoringLog(monitoringLogsLikes);
    }
    convertMonitoringLogLikesToMonitoringLog(monitoringLogLikes) {
        return monitoringLogLikes.map(this.convertMonitoringLogLikeToMonitoringLog);
    }
    convertMonitoringLogLikeToMonitoringLog(monitoringLogLike) {
        const commonMonitoringLog = {
            id: monitoringLogLike.id,
            deviceIdentityHash: monitoringLogLike.deviceIdentityHash,
            organizationUid: monitoringLogLike.organizationUid,
            type: monitoringLogLike.type,
            createdAt: new Date(monitoringLogLike.createdAt),
        };
        const dataTypes = getPropertiesDataTypes(monitoringLogLike.type);
        const isJSON = Object.values(dataTypes).some((item) => (0, dataType_1.isOfDataType)(item, dataType_1.DataType.JSON));
        if (isJSON) {
            const propertyName = (0, object_1.getObjectKeys)(dataTypes)[0];
            let parsedJSON = null;
            try {
                parsedJSON = JSON.parse(monitoringLogLike[propertyName]);
            }
            catch (e) {
                // note: error is ok when input string is empty string (i.e. bundled applet)
                debug('Cannot parse input string: %s', e.message);
            }
            return Object.assign(Object.assign({}, commonMonitoringLog), { data: parsedJSON });
        }
        else {
            const data = {};
            for (const propertyName in dataTypes) {
                const value = monitoringLogLike[propertyName];
                if ((0, dataType_1.isOfDataType)(dataTypes[propertyName], dataType_1.DataType.TIMESTAMP)) {
                    // the types are different but the result is correct
                    data[propertyName] = new Date(value);
                }
                else if (value === '' && (0, dataType_1.isOfDataType)(dataTypes[propertyName], dataType_1.DataType.NULL)) {
                    // special form of value is empty string, if the data type is NULL, it means null value.
                    data[propertyName] = null;
                }
                else if (typeof value === 'undefined' && (0, dataType_1.isOfDataType)(dataTypes[propertyName], dataType_1.DataType.MISSING)) {
                    // Add nothing because field is optional and it may be missing
                }
                else {
                    // the types are different but the result is correct
                    data[propertyName] = value;
                }
            }
            return Object.assign(Object.assign({}, commonMonitoringLog), { data });
        }
    }
}
/**
 *
 * @param log to validate
 * @param type
 * @throws Error
 */
function isMonitoringLog(log) {
    if (_.isNil(log)) {
        return false;
    }
    if (Array.isArray(log)) {
        return false;
    }
    if (!(log && typeof log === 'object' && log.constructor === Object)) {
        return false;
    }
    const tmpLog = log;
    if (!tmpLog || !tmpLog.type || !tmpLog.id || !tmpLog.createdAt || !tmpLog.deviceIdentityHash) {
        return false;
    }
    const dataTypes = getPropertiesDataTypes(tmpLog.type);
    const propNames = Object.entries(dataTypes)
        .filter(([_propName, propType]) => !(0, dataType_1.isOfDataType)(propType, dataType_1.DataType.MISSING))
        .map(([propName, _propType]) => propName);
    return propNames.every((prop) => log.hasOwnProperty(prop));
}
function getPropertiesDataTypes(type) {
    switch (type) {
        case DeviceTelemetryType_1.DeviceTelemetryType.OFFLINE_RANGE:
            return {
                since: dataType_1.DataType.TIMESTAMP,
                until: dataType_1.DataType.TIMESTAMP,
            };
        case DeviceTelemetryType_1.DeviceTelemetryType.ONLINE_STATUS:
            return {
                online: dataType_1.DataType.BOOLEAN,
            };
        case DeviceTelemetryType_1.DeviceTelemetryType.DISPLAY_SETTING:
            return {
                backlight: dataType_1.DataType.INT,
                contrast: dataType_1.DataType.INT,
                sharpness: dataType_1.DataType.INT,
                maxTemperature: dataType_1.DataType.INT,
            };
        case DeviceTelemetryType_1.DeviceTelemetryType.INPUT_SOURCE:
            return {
                inputSource: dataType_1.DataType.STRING,
            };
        case DeviceTelemetryType_1.DeviceTelemetryType.VOLUME:
            return {
                volume: dataType_1.DataType.INT,
            };
        case DeviceTelemetryType_1.DeviceTelemetryType.BRIGHTNESS:
            return {
                brightness: dataType_1.DataType.INT,
            };
        case DeviceTelemetryType_1.DeviceTelemetryType.TIMERS:
            return {
                timers: dataType_1.DataType.JSON,
            };
        case DeviceTelemetryType_1.DeviceTelemetryType.PROPRIETARY_TIMERS:
            return {
                proprietaryTimers: dataType_1.DataType.JSON,
            };
        case DeviceTelemetryType_1.DeviceTelemetryType.RESOLUTION:
            return {
                resolution: dataType_1.DataType.JSON,
            };
        case DeviceTelemetryType_1.DeviceTelemetryType.ORIENTATION:
            return {
                orientation: dataType_1.DataType.JSON,
            };
        case DeviceTelemetryType_1.DeviceTelemetryType.REMOTE_CONTROL:
            return {
                enabled: dataType_1.DataType.BOOLEAN,
            };
        case DeviceTelemetryType_1.DeviceTelemetryType.APPLICATION_VERSION:
            return {
                version: dataType_1.DataType.STRING,
                versionNumber: dataType_1.DataType.INT,
            };
        case DeviceTelemetryType_1.DeviceTelemetryType.FRONT_DISPLAY_VERSION:
            return {
                version: dataType_1.DataType.STRING,
                versionNumber: dataType_1.DataType.INT,
            };
        case DeviceTelemetryType_1.DeviceTelemetryType.FIRMWARE_VERSION:
            return {
                version: dataType_1.DataType.STRING,
            };
        case DeviceTelemetryType_1.DeviceTelemetryType.DEBUG:
            return {
                appletEnabled: dataType_1.DataType.BOOLEAN,
                nativeEnabled: dataType_1.DataType.BOOLEAN,
            };
        case DeviceTelemetryType_1.DeviceTelemetryType.DATETIME:
            return {
                datetime: dataType_1.DataType.JSON,
            };
        case DeviceTelemetryType_1.DeviceTelemetryType.POWER_ACTIONS_SCHEDULE:
            return {
                powerActionsSchedule: dataType_1.DataType.JSON,
            };
        case DeviceTelemetryType_1.DeviceTelemetryType.TEMPERATURE:
            return {
                temperature: dataType_1.DataType.FLOAT,
            };
        case DeviceTelemetryType_1.DeviceTelemetryType.BUNDLED_APPLET:
            return {
                // eslint-disable-next-line no-bitwise -- This bitwise operator is intended
                bundledApplet: dataType_1.DataType.JSON | dataType_1.DataType.NULL,
            };
        case DeviceTelemetryType_1.DeviceTelemetryType.AUTO_RECOVERY:
            return {
                enabled: dataType_1.DataType.BOOLEAN,
                // eslint-disable-next-line no-bitwise -- This bitwise operator is intended
                healthcheckIntervalMs: dataType_1.DataType.INT | dataType_1.DataType.MISSING,
                // eslint-disable-next-line no-bitwise -- This bitwise operator is intended
                autoEnableTimeoutMs: dataType_1.DataType.INT | dataType_1.DataType.MISSING,
            };
        case DeviceTelemetryType_1.DeviceTelemetryType.PEER_RECOVERY:
            return {
                enabled: dataType_1.DataType.BOOLEAN,
                // eslint-disable-next-line no-bitwise -- This bitwise operator is intended
                urlLauncherAddress: dataType_1.DataType.STRING | dataType_1.DataType.MISSING,
                // eslint-disable-next-line no-bitwise -- This bitwise operator is intended
                autoEnableTimeoutMs: dataType_1.DataType.INT | dataType_1.DataType.MISSING,
            };
        case DeviceTelemetryType_1.DeviceTelemetryType.MANAGEMENT_CAPABILITIES:
            return {
                capable: dataType_1.DataType.JSON,
            };
        case DeviceTelemetryType_1.DeviceTelemetryType.FRONT_CAPABILITIES:
            return {
                capable: dataType_1.DataType.JSON,
            };
        case DeviceTelemetryType_1.DeviceTelemetryType.PROXY:
            return {
                enabled: dataType_1.DataType.BOOLEAN,
                // eslint-disable-next-line no-bitwise -- This bitwise operator is intended
                uri: dataType_1.DataType.STRING | dataType_1.DataType.NULL,
            };
        case DeviceTelemetryType_1.DeviceTelemetryType.WIFI_STRENGTH:
            return {
                strength: dataType_1.DataType.INT,
            };
        case DeviceTelemetryType_1.DeviceTelemetryType.NETWORK_INTERFACES:
            return {
                networkInterfaces: dataType_1.DataType.JSON,
            };
        case DeviceTelemetryType_1.DeviceTelemetryType.EXTENDED_MANAGEMENT:
            return {
                // eslint-disable-next-line no-bitwise -- This bitwise operator is intended
                url: dataType_1.DataType.STRING | dataType_1.DataType.NULL,
            };
        case DeviceTelemetryType_1.DeviceTelemetryType.DISPLAY_POWER_ON:
            return {
                powerOn: dataType_1.DataType.BOOLEAN,
            };
        default:
            throw new Error(`Unknown Monitoring type: ${type}`);
    }
}
//# sourceMappingURL=helper.js.map