"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.createWsSocketServer = void 0;
const WebSocket = require("isomorphic-ws");
const events_1 = require("events");
const generator_1 = require("../Hash/generator");
const Debug = require("debug");
const http_1 = require("../Utils/http");
const debug = Debug('@signageos/lib:WebSocket:wsServerFactory');
var CloseCode;
(function (CloseCode) {
    CloseCode[CloseCode["NO_PONG"] = 3000] = "NO_PONG";
    CloseCode[CloseCode["PING_FAILED"] = 3001] = "PING_FAILED";
    CloseCode[CloseCode["FORCE_DISCONNECT"] = 3002] = "FORCE_DISCONNECT";
})(CloseCode || (CloseCode = {}));
function createWsSocketServer(httpServer, options = {}) {
    let serverClosed = false;
    const wss = new WebSocket.Server({
        server: httpServer,
        path: options.path,
    });
    const wsMap = {};
    const interval = setInterval(() => {
        Object.keys(wsMap).forEach((wsUid) => {
            if (!wsMap[wsUid].isAlive) {
                wsMap[wsUid].safeTerminate(CloseCode.NO_PONG, "Didn't receive pong");
            }
            else {
                wsMap[wsUid].isAlive = false;
                try {
                    wsMap[wsUid].ws.ping();
                }
                catch (error) {
                    wsMap[wsUid].safeTerminate(CloseCode.PING_FAILED, 'Ping error');
                }
            }
        });
    }, 30e3);
    return {
        server: {
            bindConnection(serverListener) {
                wss.on('error', (error) => console.error(error));
                wss.on('connection', (ws, req) => {
                    debug('connection', { headers: req.headers });
                    let closedPromise;
                    const wsUid = (0, generator_1.generateUniqueHash)();
                    const socketHostname = (0, http_1.getHostnameFromRequestHeaders)(req.headers);
                    function safeTerminate(code, reason) {
                        debug('Safe terminate', wsMap[wsUid]);
                        ws.terminate();
                        if (typeof wsMap[wsUid] !== 'undefined') {
                            ws.emit('close', code, reason);
                        }
                    }
                    wsMap[wsUid] = {
                        ws,
                        isAlive: true,
                        safeTerminate,
                    };
                    ws.on('pong', () => {
                        debug('pong');
                        wsMap[wsUid].isAlive = true;
                    });
                    ws.once('close', () => {
                        debug('close');
                        closedPromise = Promise.resolve();
                        delete wsMap[wsUid];
                    });
                    const messageEmitter = new events_1.EventEmitter();
                    const responseEmitter = new events_1.EventEmitter();
                    ws.on('message', (messageData) => {
                        debug('message', messageData);
                        const message = JSON.parse(messageData);
                        switch (message.type) {
                            case 'response':
                                responseEmitter.emit(message.responseUid);
                                break;
                            case 'request':
                                messageEmitter.emit(message.event, message.payload);
                                if (message.responseUid) {
                                    const responseMessage = {
                                        type: 'response',
                                        responseUid: message.responseUid,
                                    };
                                    try {
                                        ws.send(JSON.stringify(responseMessage));
                                    }
                                    catch (error) {
                                        debug('Cannot send response back', message, error);
                                    }
                                }
                                break;
                            case 'ping':
                                try {
                                    const pongMessage = { type: 'pong' };
                                    ws.send(JSON.stringify(pongMessage));
                                }
                                catch (error) {
                                    debug('Cannot send pong', error);
                                }
                                break;
                            default:
                        }
                    });
                    ws.on('error', (error) => console.error(error));
                    serverListener({
                        getDisconnectedPromise() {
                            return new Promise((resolve) => {
                                ws.once('close', (code, reason) => resolve({ code, reason }));
                            });
                        },
                        bindError(listener) {
                            ws.on('error', listener);
                            return () => ws.removeListener('error', listener);
                        },
                        bindMessage(event, listener) {
                            messageEmitter.on(event, listener);
                            return () => messageEmitter.removeListener(event, listener);
                        },
                        getMessagePromise(event) {
                            return new Promise((resolve) => {
                                messageEmitter.once(event, resolve);
                            });
                        },
                        sendMessage(event, payload) {
                            const message = {
                                type: 'request',
                                event,
                                payload,
                            };
                            return new Promise((resolve, reject) => {
                                try {
                                    ws.send(JSON.stringify(message), (error) => (error ? reject(error) : resolve()));
                                }
                                catch (error) {
                                    reject(error);
                                }
                            });
                        },
                        sendMessageExpectingResponse(event, payload) {
                            return __awaiter(this, void 0, void 0, function* () {
                                const responseUid = (0, generator_1.generateUniqueHash)();
                                const message = {
                                    type: 'request',
                                    event,
                                    payload,
                                    responseUid,
                                };
                                const responsePromise = new Promise((resolve) => {
                                    responseEmitter.once(responseUid, () => resolve());
                                });
                                const sendPromise = new Promise((resolve, reject) => {
                                    try {
                                        ws.send(JSON.stringify(message), (error) => (error ? reject(error) : resolve()));
                                    }
                                    catch (error) {
                                        reject(error);
                                    }
                                });
                                yield Promise.all([sendPromise, responsePromise]);
                            });
                        },
                        disconnect(code, reason) {
                            if (typeof closedPromise !== 'undefined') {
                                return closedPromise;
                            }
                            closedPromise = new Promise((resolve) => {
                                const terminateTimeout = setTimeout(() => {
                                    safeTerminate(CloseCode.FORCE_DISCONNECT, 'Force disconnect');
                                    resolve();
                                }, 30e3);
                                ws.once('close', () => {
                                    clearTimeout(terminateTimeout);
                                    resolve();
                                });
                            });
                            ws.close(code, reason);
                            return closedPromise;
                        },
                        drain() {
                            ws.removeAllListeners();
                            messageEmitter.removeAllListeners();
                            responseEmitter.removeAllListeners();
                        },
                        hostname: socketHostname,
                        path: req.url,
                        headers: req.headers,
                    });
                });
            },
        },
        listen() {
            return __awaiter(this, void 0, void 0, function* () {
                debug('Automatically started');
            });
        },
        close() {
            return __awaiter(this, void 0, void 0, function* () {
                if (serverClosed) {
                    throw new Error('Server was already closed');
                }
                serverClosed = true;
                clearInterval(interval);
                yield new Promise((resolve) => {
                    wss.close(resolve);
                });
            });
        },
    };
}
exports.createWsSocketServer = createWsSocketServer;
//# sourceMappingURL=wsServerFactory.js.map