"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.MongoDBConnection = void 0;
const mongoose = require("mongoose");
const Debug = require("debug");
const proprietaryCluster_1 = require("./proprietaryCluster");
const genericGracefulExit_1 = require("../Events/genericGracefulExit");
mongoose.Promise = Promise;
const debug = Debug('@signageos/lib:MongoDB:mongoDBConnectionFactory');
const defaultOptions = {
    autoIndex: false,
    connectTimeoutMS: 10 * 60 * 1e3,
    socketTimeoutMS: 3 * 60 * 60 * 1e3,
    monitorCommands: true,
};
class MongoDBConnection {
    constructor(mongoDbDsn, options = {}, onExit) {
        this.mongoDbDsn = mongoDbDsn;
        this.connection = null;
        this.metrics = null;
        this.options = Object.assign(Object.assign({}, defaultOptions), options);
        const reconnectTimeout = 30e3;
        this.gracefulExit = new genericGracefulExit_1.GenericGracefulExit('mongo', reconnectTimeout, onExit);
    }
    connect() {
        return __awaiter(this, void 0, void 0, function* () {
            debug('connect');
            const updatedMongDbDsn = yield (0, proprietaryCluster_1.updateDsnIfProprietaryCluster)(this.mongoDbDsn);
            const connection = mongoose.createConnection(updatedMongDbDsn, this.options);
            this.applyGracefulExit(connection);
            if (this.metrics) {
                this.monitorConnection(connection, this.metrics.exporter, this.metrics.connectionName);
            }
            yield connection.asPromise();
            debug('connected');
            this.connection = connection;
        });
    }
    close() {
        return __awaiter(this, void 0, void 0, function* () {
            if (this.connection) {
                yield this.connection.close();
            }
        });
    }
    monitor(exporter, connectionName) {
        if (this.connection) {
            debug(`monitor connection ${connectionName} now`);
            this.monitorConnection(this.connection, exporter, connectionName);
        }
        else {
            debug(`monitor connection ${connectionName}, once it connects`);
            // if connection is not yet established, store the exporter and name for later use
            this.metrics = {
                exporter,
                connectionName,
            };
        }
    }
    applyGracefulExit(connection) {
        connection.on('disconnected', () => __awaiter(this, void 0, void 0, function* () {
            yield this.gracefulExit.waitToConnectOrDie();
        }));
        connection.on('error', (_err) => __awaiter(this, void 0, void 0, function* () {
            yield this.gracefulExit.waitToConnectOrDie();
        }));
        connection.on('connected', () => __awaiter(this, void 0, void 0, function* () {
            this.gracefulExit.connected();
        }));
        connection.on('close', () => __awaiter(this, void 0, void 0, function* () {
            this.gracefulExit.closed();
        }));
    }
    /**
     * @see https://www.mongodb.com/docs/drivers/node/current/fundamentals/monitoring/command-monitoring/#command-monitoring
     */
    monitorConnection(connection, exporter, connectionName) {
        debug(`monitor connection ${connectionName}`);
        const mongoClient = connection.getClient();
        exporter.monitorMongoClient(mongoClient, connectionName);
        const eventNames = ['commandStarted', 'commandSucceeded', 'commandFailed'];
        for (const eventName of eventNames) {
            mongoClient.addListener(eventName, (event) => {
                debug(eventName, 
                // https://github.com/GoogleChromeLabs/jsbi/issues/30
                JSON.stringify(event, (_key, value) => (typeof value === 'bigint' ? value.toString() : value)));
            });
        }
    }
}
exports.MongoDBConnection = MongoDBConnection;
//# sourceMappingURL=MongoDBConnection.js.map