"use strict";
var __rest = (this && this.__rest) || function (s, e) {
    var t = {};
    for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p) && e.indexOf(p) < 0)
        t[p] = s[p];
    if (s != null && typeof Object.getOwnPropertySymbols === "function")
        for (var i = 0, p = Object.getOwnPropertySymbols(s); i < p.length; i++) {
            if (e.indexOf(p[i]) < 0 && Object.prototype.propertyIsEnumerable.call(s, p[i]))
                t[p[i]] = s[p[i]];
        }
    return t;
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.redirectAllLogsToLogger = exports.logger = void 0;
const winston = require("winston");
const logger_1 = require("./logger");
const console_1 = require("./console");
const debug_1 = require("./debug");
var LogFormat;
(function (LogFormat) {
    LogFormat["JSON"] = "json";
    LogFormat["JSON_PRETTY"] = "json_pretty";
    LogFormat["Simple"] = "simple";
})(LogFormat || (LogFormat = {}));
const logLevel = process.env.LOG_LEVEL || 'debug';
const logFormat = process.env.LOG_FORMAT || LogFormat.JSON;
const ensureJSONKeyOrder = winston.format((info) => {
    const { level, message } = info, rest = __rest(info, ["level", "message"]);
    return Object.assign({ level, message }, rest);
});
const winstonFormatters = [
    winston.format.errors({ stack: true }), // parses errors and adds stack trace
];
switch (logFormat) {
    case LogFormat.JSON:
        winstonFormatters.push(ensureJSONKeyOrder(), // ensures that the keys in the JSON are always in the order we want
        winston.format.json({ deterministic: false }));
        break;
    case LogFormat.JSON_PRETTY:
        winstonFormatters.push(ensureJSONKeyOrder(), // ensures that the keys in the JSON are always in the order we want
        winston.format.json(), winston.format.prettyPrint({ colorize: true }));
        break;
    case LogFormat.Simple:
        winstonFormatters.push(winston.format.colorize(), // adds colors to the log
        winston.format.simple());
        break;
    default:
        throw new Error(`Unknown log format: ${logFormat}, allowed values are: json, simple`);
}
/**
 * Since we aim to have a single API for all logs in the application, it's implemented as singleton.
 * If that turns to be insufficient, we can change that later.
 */
const winstonLogger = winston.createLogger({
    level: logLevel,
    format: winston.format.combine(...winstonFormatters),
    transports: [new winston.transports.Console()],
});
/**
 * Singleton logger to be used across the entire application
 */
exports.logger = new logger_1.WinstonLogger(winstonLogger);
/**
 * This function will redirect all console logs and debug logs to the logger.
 *
 * The goal is to collect all logs that are not logged through the central logger and make them go through the logger.
 * That way we can control the format and destination of all logs no matter where they are logged.
 */
function redirectAllLogsToLogger() {
    (0, console_1.redirectConsoleToLogger)(exports.logger);
    (0, debug_1.redirectDebugToLogger)(exports.logger);
}
exports.redirectAllLogsToLogger = redirectAllLogsToLogger;
//# sourceMappingURL=logging.js.map