"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.parseServiceConfig = exports.parseBasicConfig = void 0;
const lodash_1 = require("lodash");
const Debug = require("debug");
const detector_1 = require("../Path/detector");
const fs = require("fs");
const dotenv = require("dotenv");
const environment_1 = require("./environment");
const validation_1 = require("./validation");
const debug = Debug('@signageos/lib:Config:config');
function readJsonFile(jsonPath) {
    return JSON.parse(fs.readFileSync(jsonPath).toString());
}
/**
 * Parses environment variables and package.json into a config object for generic applications
 *
 * @param args - optionally not load process.env for testing purposes, leave undefined in application code
 * @returns basic config object that can be used by any NodeJS application
 */
function parseBasicConfig(args) {
    var _a, _b;
    const environment = (_a = process.env.NODE_ENV) !== null && _a !== void 0 ? _a : environment_1.NodeEnvironment.Dev;
    const addToProcessEnv = (args === null || args === void 0 ? void 0 : args.addToProcessEnv) === false ? false : true;
    const rootPath = (0, detector_1.findFirstParentModulePath)((_b = args === null || args === void 0 ? void 0 : args.basePath) !== null && _b !== void 0 ? _b : process.cwd());
    const packageConfig = validation_1.packageConfigSchema.parse(loadPackageConfig(rootPath));
    const appName = packageConfig.name.replace(/^@[^/]+\//, ''); // replace prefix @<namespace>/
    try {
        const localEnv = readJsonFile(rootPath + '/config/env.' + environment + '.json');
        if (addToProcessEnv) {
            dotenv.populate(process.env, localEnv, { override: false });
        }
    }
    catch (err) {
        console.info(`Not using overriding env.${environment}.json file`);
    }
    const parsedEnv = validation_1.envVarsBasicConfigSchema.parse(process.env);
    const config = {
        environment,
        paths: {
            rootPath: rootPath !== null && rootPath !== void 0 ? rootPath : '..',
            configPath: rootPath + '/config',
            testsPath: rootPath + '/tests',
            toolsPath: rootPath + '/tools',
            distPath: rootPath + '/dist',
        },
        app: {
            /**
             * Parsed out of package name in package.json
             * @example @signageos/api -> api
             */
            name: appName,
            version: packageConfig.version,
        },
        npm: packageConfig,
        sentry: {
            enabled: parsedEnv.sentry_enabled,
            config: {
                dsn: parsedEnv.sentry_dsn,
                release: packageConfig.version,
                environment: parsedEnv.NODE_ENV,
                tags: {
                    application: appName,
                    git_commit: packageConfig.gitHead || '',
                },
            },
        },
    };
    debug('parsed basic config', config);
    return config;
}
exports.parseBasicConfig = parseBasicConfig;
/**
 * Parses environment variables and package.json into a config object for services
 *
 * Extends parseBasicConfig to include env vars for services
 *
 * @param args - optionally not load process.env for testing purposes, leave undefined in application code
 * @returns basic config object that can be used by any NodeJS application
 */
function parseServiceConfig(args) {
    const parsedEnv = validation_1.envVarsServiceConfigSchema.parse(process.env);
    const baseConfig = parseBasicConfig(args);
    const serviceConfig = {
        app: {
            port: parsedEnv.PORT,
            monitoringPort: parsedEnv.MONITORING_PORT,
        },
    };
    return (0, lodash_1.merge)({}, baseConfig, serviceConfig);
}
exports.parseServiceConfig = parseServiceConfig;
function loadPackageConfig(rootPath) {
    const packageConfigPath = rootPath + '/package.json';
    const packageConfig = readJsonFile(packageConfigPath);
    // TODO replace with link to wiki on the web, once it's available
    const troubleshootingLinkMessage = 'See https://gitlab.com/signageos/wiki/-/blob/master/docs/basics/how-to/services/how-to-handle-env-and-config.md#packagejson-is-missing-name-or-version';
    if (!packageConfig.name) {
        throw new Error('package.json is missing name. ' + troubleshootingLinkMessage);
    }
    if (!packageConfig.version) {
        throw new Error('package.json is missing version. ' + troubleshootingLinkMessage);
    }
    return packageConfig;
}
//# sourceMappingURL=config.js.map