/**
 * @returns wait() A progressive wait function.
 * @returns reset() A function that resets the timeout to the initial value.
 * @returns getCurrentTimeoutMs() How long it's currently waiting since the last wait.
 *     It's not changing during a single iteration of wait().
 * @returns getNextTimeoutMs() Same as the getCurrentTimeoutMs but for upcoming wait() iteration.
 *     It's getCurrentTimeoutMs() value multiplied with initial multiplier (respecting the maximum value)
 */
export interface IProgressiveWait {
    wait(): Promise<void>;
    reset(): void;
    getCurrentTimeoutMs(): number;
    getNextTimeoutMs(): number;
}
export interface IProgressiveWaitFactory {
    (initialTimeoutMs: number): IProgressiveWait;
}
/**
 * Creates a progressive wait function that can be used to wait for a period of time,
 * and then progressively increase the wait time by a multiplier.
 *
 * @param initialTimeoutMs The initial timeout in milliseconds.
 * @param progressiveMultiplier The multiplier to use when increasing the timeout.
 * @param maxTimeoutMs The maximum timeout in milliseconds.
 * @example of usage:
 * const wait = createProgressiveWait(1000);
 * await wait(); // waits for 1000ms
 * await wait(); // waits for 2000ms
 * await wait(); // waits for 4000ms
 * await wait(); // waits for 8000ms
 * await wait(); // waits for 10000ms
 *
 * const wait = createProgressiveWait(1000, 3);
 * await wait(); // waits for 1000ms
 * await wait(); // waits for 3000ms
 * await wait(); // waits for 9000ms
 * await wait(); // waits for 10000ms
 */
export declare const createProgressiveWait: (initialTimeoutMs: number, progressiveMultiplier?: number, maxTimeoutMs?: number) => IProgressiveWait;
