"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.InvalidApplicationTypeField = exports.MissingRequiredValueField = exports.InvalidPolicyItem = exports.InvalidPolicyItems = void 0;
exports.validateAllPolicyItems = validateAllPolicyItems;
exports.default = validatePolicyItem;
exports.isPolicyItemMissingRequired = isPolicyItemMissingRequired;
exports.isValidPolicyItemApplicationType = isValidPolicyItemApplicationType;
const isValidApplicationType_1 = require("../../Application/Validation/isValidApplicationType");
const DeviceSettingsType_1 = require("../../Device/Settings/DeviceSettingsType");
const validateApplicationVersionSettings_1 = require("../../Device/Settings/Validation/validateApplicationVersionSettings");
const validateFirmwareVersionSettings_1 = require("../../Device/Settings/Validation/validateFirmwareVersionSettings");
const validateOrientationSettings_1 = require("../../Device/Settings/Validation/validateOrientationSettings");
const validateRemoteControlSettings_1 = require("../../Device/Settings/Validation/validateRemoteControlSettings");
const validateResolutionSettings_1 = require("../../Device/Settings/Validation/validateResolutionSettings");
const validateVolumeSettings_1 = require("../../Device/Settings/Validation/validateVolumeSettings");
const Validation_1 = require("../../Validation/Validation");
const validateBrightnessSettingsPolicy_1 = require("./validateBrightnessSettingsPolicy");
const validateTimerSettingsPolicy_1 = require("./validateTimerSettingsPolicy");
const validateDebugSettings_1 = require("../../Device/Settings/Validation/validateDebugSettings");
const validateDateTimeSettings_1 = require("../../Device/Settings/Validation/validateDateTimeSettings");
const validatePowerActionScheduleSettingsPolicy_1 = require("./validatePowerActionScheduleSettingsPolicy");
const validateTemperaturePolicy_1 = require("./validateTemperaturePolicy");
const validateAutoRecoverySettings_1 = require("../../Device/Settings/Validation/validateAutoRecoverySettings");
const validatePeerRecoverySettings_1 = require("../../Device/Settings/Validation/validatePeerRecoverySettings");
const validateConnectionMethodSettings_1 = require("../../Device/Settings/Validation/validateConnectionMethodSettings");
const validateAppletPolicy_1 = require("./validateAppletPolicy");
const validateExtendedManagementPolicy_1 = require("./validateExtendedManagementPolicy");
function validateAllPolicyItems(policyItems) {
    const validation = (0, Validation_1.createValidation)(policyItems);
    if (!Array.isArray(policyItems)) {
        validation.addError(new InvalidPolicyItems(policyItems));
    }
    else {
        policyItems.forEach((policyItem) => {
            const itemValidation = validatePolicyItem(policyItem);
            validation.addError(...itemValidation.getErrors());
        });
    }
    return validation.getOutput();
}
function validatePolicyItem(policyItem) {
    var _a;
    const { type } = (_a = policyItem) !== null && _a !== void 0 ? _a : {};
    const use = useValidation(policyItem);
    switch (type) {
        case DeviceSettingsType_1.DeviceSettingsType.VOLUME:
            return use(validateVolumeSettings_1.default);
        case DeviceSettingsType_1.DeviceSettingsType.BRIGHTNESS:
            return use(validateBrightnessSettingsPolicy_1.default);
        case DeviceSettingsType_1.DeviceSettingsType.TIMERS:
            return use(validateTimerSettingsPolicy_1.default);
        case DeviceSettingsType_1.DeviceSettingsType.PROPRIETARY_TIMERS:
            return use(validateTimerSettingsPolicy_1.validateProprietaryTimerSettingsPolicy);
        case DeviceSettingsType_1.DeviceSettingsType.RESOLUTION:
            return use(validateResolutionSettings_1.default);
        case DeviceSettingsType_1.DeviceSettingsType.ORIENTATION:
            return use(validateOrientationSettings_1.default);
        case DeviceSettingsType_1.DeviceSettingsType.REMOTE_CONTROL:
            return use(validateRemoteControlSettings_1.default);
        case DeviceSettingsType_1.DeviceSettingsType.APPLICATION_VERSION:
            return use(validateApplicationVersionSettings_1.default);
        case DeviceSettingsType_1.DeviceSettingsType.FIRMWARE_VERSION:
            return use(validateFirmwareVersionSettings_1.default);
        case DeviceSettingsType_1.DeviceSettingsType.DEBUG:
            return use(validateDebugSettings_1.default);
        case DeviceSettingsType_1.DeviceSettingsType.DATETIME:
            return use(validateDateTimeSettings_1.default);
        case DeviceSettingsType_1.DeviceSettingsType.POWER_ACTIONS_SCHEDULE:
            return use(validatePowerActionScheduleSettingsPolicy_1.default);
        case DeviceSettingsType_1.DeviceSettingsType.TEMPERATURE:
            return use(validateTemperaturePolicy_1.default);
        case DeviceSettingsType_1.DeviceSettingsType.AUTO_RECOVERY:
            return use(validateAutoRecoverySettings_1.default);
        case DeviceSettingsType_1.DeviceSettingsType.PEER_RECOVERY:
            return use(validatePeerRecoverySettings_1.default);
        case DeviceSettingsType_1.DeviceSettingsType.CONNECTION_METHOD:
            return use(validateConnectionMethodSettings_1.default);
        case DeviceSettingsType_1.DeviceSettingsType.APPLET:
            return use(validateAppletPolicy_1.default);
        case DeviceSettingsType_1.DeviceSettingsType.EXTENDED_MANAGEMENT:
            return use(validateExtendedManagementPolicy_1.default);
        default:
            return createInvalidPolicyItemValidation(policyItem);
    }
}
function useValidation(policyItem) {
    return (validateValue) => validatePolicyItemBase(policyItem, validateValue);
}
function validatePolicyItemBase(policyItem, validateValue) {
    var _a;
    const { value, applicationType } = (_a = policyItem) !== null && _a !== void 0 ? _a : {};
    const isMissingRequired = isPolicyItemMissingRequired(policyItem);
    const isApplicationTypeValid = isValidPolicyItemApplicationType(applicationType);
    const valueValidation = validateValue(value);
    const getValueCast = () => valueValidation.getCast();
    const castPolicyItem = (policyItem) => ({
        type: policyItem.type,
        applicationType: policyItem.applicationType,
        value: getValueCast(),
    });
    const validation = (0, Validation_1.createValidation)(policyItem, castPolicyItem);
    if (isMissingRequired) {
        validation.addError(new MissingRequiredValueField(value));
    }
    else {
        if (!isApplicationTypeValid) {
            validation.addError(new InvalidApplicationTypeField(applicationType));
        }
        validation.addError(...valueValidation.getErrors());
    }
    return validation.getOutput();
}
function createInvalidPolicyItemValidation(policyItem) {
    const validation = (0, Validation_1.createValidation)({});
    validation.addError(new InvalidPolicyItem(policyItem));
    return validation.getOutput();
}
function isPolicyItemMissingRequired(policyItem) {
    var _a;
    const { type, value } = (_a = policyItem) !== null && _a !== void 0 ? _a : {};
    const isMissingRequired = type === undefined || value === undefined;
    return isMissingRequired;
}
function isValidPolicyItemApplicationType(type) {
    const isValidType = type === undefined || (0, isValidApplicationType_1.default)(type);
    return isValidType;
}
class InvalidPolicyItems extends Error {
    constructor(policyItems) {
        super(`Policy Items is not array: ${JSON.stringify(policyItems)}`);
    }
}
exports.InvalidPolicyItems = InvalidPolicyItems;
class InvalidPolicyItem extends Error {
    constructor(policyItem) {
        super(`Policy Item Object is invalid, needs type field to continue evaluation: ${JSON.stringify(policyItem)}`);
    }
}
exports.InvalidPolicyItem = InvalidPolicyItem;
class MissingRequiredValueField extends Error {
    constructor(value) {
        super(`Policy Item is missing required value field: ${JSON.stringify(value)}`);
    }
}
exports.MissingRequiredValueField = MissingRequiredValueField;
class InvalidApplicationTypeField extends Error {
    constructor(type) {
        super(`Policy Item Application Type is not valid: ${JSON.stringify(type)}`);
    }
}
exports.InvalidApplicationTypeField = InvalidApplicationTypeField;
