"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NetworkInterfacesSchema = exports.NetworkInterfaceSchema = exports.NetworkInterfaceType = exports.MacAddressSchema = void 0;
const zod_1 = require("zod");
exports.MacAddressSchema = zod_1.z
    .string()
    .refine((value) => {
    const macAddressRegex = /^([0-9A-Fa-f]{2}([:-]| )?){6}$/;
    return macAddressRegex.test(value);
}, 'Mac address must be in the format XX:XX:XX:XX:XX:XX where X is a hexadecimal character')
    .transform((value) => {
    return value
        .replace(/[:-\s]/g, '')
        .replace(/(.{2})/g, '$1:')
        .slice(0, -1)
        .toLowerCase();
});
/**
 * See also:
 * * https://www.thomas-krenn.com/en/wiki/Predictable_Network_Interface_Names.
 * * https://openwrt.org/docs/guide-developer/networking/network.interfaces
 */
exports.NetworkInterfaceType = [
    /** Wireless LAN. `wl*`. */
    'wifi',
    /** Ethernet. `en*`, `eth*`. */
    'ethernet',
    /** VPN or Tunnel Interface. `ppp*`, `vpn*`, `tun*`. */
    'vpn',
    /** Wireless WAN. `ww*`. */
    'mobile',
];
const BaseNetworkInterfaceSchema = zod_1.z.object({
    name: zod_1.z.string().max(255),
    macAddress: exports.MacAddressSchema,
    disabled: zod_1.z.boolean().optional(),
    gateway: zod_1.z.string().optional(),
    localAddress: zod_1.z.string().optional(),
    netmask: zod_1.z.string().optional(),
    dns: zod_1.z.array(zod_1.z.string()).optional(),
});
const WifiNetworkInterfaceSchema = BaseNetworkInterfaceSchema.extend({
    type: zod_1.z.literal('wifi'),
    wifiStrength: zod_1.z.number().optional(),
    wifiSsid: zod_1.z.string().optional(),
});
const EthernetNetworkInterfaceSchema = BaseNetworkInterfaceSchema.extend({
    type: zod_1.z.literal('ethernet'),
});
const VpnNetworkInterfaceSchema = BaseNetworkInterfaceSchema.extend({
    type: zod_1.z.literal('vpn'),
});
const MobileNetworkInterfaceSchema = BaseNetworkInterfaceSchema.extend({
    type: zod_1.z.literal('mobile'),
});
exports.NetworkInterfaceSchema = zod_1.z.discriminatedUnion('type', [
    WifiNetworkInterfaceSchema,
    EthernetNetworkInterfaceSchema,
    VpnNetworkInterfaceSchema,
    MobileNetworkInterfaceSchema,
]);
/** @deprecated use singular @type {NetworkInterfaceSchema} */
exports.NetworkInterfacesSchema = exports.NetworkInterfaceSchema;
