"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.InvalidConditionToAlertType = exports.InvalidCondition = exports.InvalidNumericRangeCondition = exports.InvalidStringOccurencesInTimeRangeCondition = exports.InvalidOccurrencesInTimeRangeCondition = exports.InvalidBooleanCondition = exports.inputSource = exports.wifiEthernet = void 0;
exports.default = isAlertConditionValid;
const AlertCondition_1 = require("../AlertCondition");
const validateAlertType_1 = require("./validateAlertType");
const validateResolutionSettings_1 = require("../../Device/Settings/Validation/validateResolutionSettings");
const isValidNumericOperator_1 = require("../../Validation/isValidNumericOperator");
const Validation_1 = require("../../Validation/Validation");
const isValidVideoOrientation_1 = require("../../Device/Resolution/Validation/isValidVideoOrientation");
const DeviceSettingsType_1 = require("../../Device/Settings/DeviceSettingsType");
const isValidOrientation_1 = require("../../Device/Resolution/Validation/isValidOrientation");
const validatePowerActionsScheduled_1 = require("./validatePowerActionsScheduled");
const validateTimerSettings_1 = require("../../Device/Settings/Validation/validateTimerSettings");
exports.wifiEthernet = ['wifi', 'ethernet'];
exports.inputSource = ['urlLauncher', 'hdmi1', 'hdmi2', 'hdmi3', 'hdmi4'];
function isAlertConditionValid(conditions, alertType) {
    const validation = (0, Validation_1.createValidation)({});
    if (Array.isArray(conditions)) {
        const isValidAlertType = (0, validateAlertType_1.default)(alertType);
        if (isValidAlertType.hasErrors()) {
            validation.addError(...isValidAlertType.getErrors());
        }
        const validAlertType = alertType;
        const validationConditions = conditions;
        switch (validAlertType) {
            case 'DEVICE': {
                const areValidConditionTypes = validationConditions.every((condition) => condition.type in AlertCondition_1.DeviceConditionType);
                if (areValidConditionTypes) {
                    validationConditions.every((condition) => isValidConditionOp(condition, validation));
                }
                else {
                    validation.addError(new InvalidConditionToAlertType(validationConditions, validAlertType));
                }
                break;
            }
            case 'POLICY': {
                const areValidPolicyTypes = validationConditions.every((condition) => condition.type in AlertCondition_1.PolicyConditionType);
                if (areValidPolicyTypes) {
                    validationConditions.every((condition) => isValidConditionOp(condition, validation));
                }
                else {
                    validation.addError(new InvalidConditionToAlertType(validationConditions, alertType));
                }
                break;
            }
            case 'APPLET': {
                const areValidAppletTypes = validationConditions.every((condition) => condition.type in AlertCondition_1.AppletConditionType);
                if (areValidAppletTypes) {
                    validationConditions.every((condition) => isValidConditionOp(condition.op, validation));
                }
                else {
                    validation.addError(new InvalidConditionToAlertType(validationConditions, alertType));
                }
                break;
            }
        }
    }
    else {
        validation.addError(new InvalidConditionToAlertType(conditions, alertType));
    }
    return validation.getOutput();
}
function isValidConditionOp(kindCondition, validation) {
    const isValidKind = kindCondition;
    const condition = kindCondition;
    if (isValidKind) {
        switch (kindOfCondition(condition.type)) {
            case AlertCondition_1.ConditionKind.BOOLEAN: {
                const isValidCondition = typeof condition.op === 'boolean';
                if (!isValidCondition) {
                    validation.addError(new InvalidBooleanCondition(condition));
                }
                break;
            }
            case AlertCondition_1.ConditionKind.OCCURRENCES_IN_TIME_RANGE_TO_PAST: {
                const isValidCondition = Array.isArray(condition.op) &&
                    !(0, isValidNumericOperator_1.default)(condition.op[0]).hasErrors() &&
                    typeof condition.op[1] === 'number' &&
                    typeof condition.op[2] === 'number';
                if (!isValidCondition) {
                    validation.addError(new InvalidOccurrencesInTimeRangeCondition(condition));
                }
                break;
            }
            case AlertCondition_1.ConditionKind.STRING_OCCURRENCES_IN_TIME_RANGE_TO_PAST: {
                const isValidCondition = Array.isArray(condition.op) &&
                    typeof condition.op[0] === 'string' &&
                    !(0, isValidNumericOperator_1.default)(condition.op[1]).getErrors() &&
                    typeof condition.op[2] === 'number' &&
                    typeof condition.op[3] === 'number' &&
                    condition.op.length === 3;
                if (!isValidCondition) {
                    validation.addError(new InvalidStringOccurencesInTimeRangeCondition(condition));
                }
                break;
            }
            case AlertCondition_1.ConditionKind.TIME_RANGE_TO_PAST: {
                const isValidCondition = validateNumericAndNumber(condition.op, 'number');
                if (!isValidCondition) {
                    validation.addError(new InvalidNumericRangeCondition(condition));
                }
                break;
            }
            case AlertCondition_1.ConditionKind.PERCENTAGE_RANGE: {
                const isValidCondition = validateNumericAndNumber(condition.op, 'number');
                if (!isValidCondition) {
                    validation.addError(new InvalidNumericRangeCondition(kindCondition));
                }
                break;
            }
            case AlertCondition_1.ConditionKind.NUMERIC_RANGE: {
                const isValidCondition = validateNumericAndNumber(condition.op, 'number');
                if (!isValidCondition) {
                    validation.addError(new InvalidNumericRangeCondition(condition));
                }
                break;
            }
            case AlertCondition_1.ConditionKind.MATCH_STRING: {
                if (typeof condition.op !== 'string') {
                    validation.addError(new InvalidCondition(kindCondition));
                }
                break;
            }
            case AlertCondition_1.ConditionKind.MATCH_SEMVER: {
                const isValidCondition = validateNumericAndNumber(condition.op, 'string');
                if (!isValidCondition) {
                    validation.addError(new InvalidNumericRangeCondition(condition));
                }
                break;
            }
            case AlertCondition_1.ConditionKind.STRING_LIST: {
                const isValidCondition = Array.isArray(condition.op) && condition.op.every((element) => typeof element === 'string');
                if (!isValidCondition) {
                    validation.addError(new InvalidCondition(condition));
                }
                break;
            }
            case AlertCondition_1.ConditionKind.POLICY_LIST: {
                const isValidCondition = Array.isArray(condition.op) && condition.op.every((condition) => condition in DeviceSettingsType_1.DeviceSettingsType);
                if (!isValidCondition) {
                    validation.addError(new InvalidCondition(condition));
                }
                break;
            }
            case AlertCondition_1.ConditionKind.WIFI_OR_ETHERNET: {
                if (!exports.wifiEthernet.includes(condition.op)) {
                    validation.addError(new InvalidCondition(condition));
                }
                break;
            }
            case AlertCondition_1.ConditionKind.INPUT_SOURCE: {
                if (!exports.inputSource.includes(condition.op)) {
                    validation.addError(new InvalidCondition(condition));
                }
                break;
            }
            case AlertCondition_1.ConditionKind.TIMERS: {
                const isValidCondition = (0, validateTimerSettings_1.default)(condition.op);
                if (isValidCondition.hasErrors()) {
                    validation.addError(...isValidCondition.getErrors());
                }
                break;
            }
            case AlertCondition_1.ConditionKind.PROPRIETARY_TIMERS: {
                const isValidCondition = (0, validateTimerSettings_1.validateProprietaryTimerSettings)(condition.op);
                if (isValidCondition.hasErrors()) {
                    validation.addError(...isValidCondition.getErrors());
                }
                break;
            }
            case AlertCondition_1.ConditionKind.RESOLUTION: {
                const isValidCondition = (0, validateResolutionSettings_1.default)(condition.op);
                if (isValidCondition.hasErrors()) {
                    validation.addError(...isValidCondition.getErrors());
                }
                break;
            }
            case AlertCondition_1.ConditionKind.ORIENTATION: {
                const isValidCondition = (0, isValidOrientation_1.default)(condition.op.orientation) &&
                    (typeof condition.op.videoOrientation === 'undefined' || (0, isValidVideoOrientation_1.default)(condition.op.videoOrientation));
                if (!isValidCondition) {
                    validation.addError(new InvalidCondition(condition));
                }
                break;
            }
            case AlertCondition_1.ConditionKind.SCHEDULED_POWER_ACTIONS: {
                const isValidCondition = (0, validatePowerActionsScheduled_1.default)(condition.op);
                if (isValidCondition.hasErrors()) {
                    validation.addError(...isValidCondition.getErrors());
                }
                break;
            }
        }
    }
    else {
        validation.addError(new InvalidCondition(JSON.stringify(condition)));
    }
    return validation.getOutput();
}
function validateNumericAndNumber(condition, match) {
    return Array.isArray(condition) && !(0, isValidNumericOperator_1.default)(condition[0]).hasErrors() && typeof condition[1] === match;
}
function kindOfCondition(conditionType) {
    let conditionKind;
    switch (conditionType) {
        case 'INCORRECT_TIME': {
            return (conditionKind = AlertCondition_1.ConditionKind.BOOLEAN);
        }
        case 'INVALID_SSL_CERTIFICATE': {
            return (conditionKind = AlertCondition_1.ConditionKind.BOOLEAN);
        }
        case 'FAILED_ACTIONS': {
            return (conditionKind = AlertCondition_1.ConditionKind.OCCURRENCES_IN_TIME_RANGE_TO_PAST);
        }
        case 'CONNECTIONS': {
            return (conditionKind = AlertCondition_1.ConditionKind.BOOLEAN);
        }
        case 'MODEL': {
            return (conditionKind = AlertCondition_1.ConditionKind.BOOLEAN);
        }
        case 'SERIAL_NUMBER': {
            return (conditionKind = AlertCondition_1.ConditionKind.BOOLEAN);
        }
        case 'NAME': {
            return (conditionKind = AlertCondition_1.ConditionKind.MATCH_STRING);
        }
        case 'PIN_CODE': {
            return (conditionKind = AlertCondition_1.ConditionKind.BOOLEAN);
        }
        case 'MANAGEMENT_PACKAGE_VERSION': {
            return (conditionKind = AlertCondition_1.ConditionKind.MATCH_SEMVER);
        }
        case 'FRONT_PACKAGE_VERSION': {
            return (conditionKind = AlertCondition_1.ConditionKind.MATCH_SEMVER);
        }
        case 'FRONT_DISPLAY_VERSION': {
            return (conditionKind = AlertCondition_1.ConditionKind.MATCH_SEMVER);
        }
        case 'FIRMWARE_TYPE': {
            return (conditionKind = AlertCondition_1.ConditionKind.MATCH_STRING);
        }
        case 'TAG_UIDS': {
            return (conditionKind = AlertCondition_1.ConditionKind.STRING_LIST);
        }
        case 'POLICIES': {
            return (conditionKind = AlertCondition_1.ConditionKind.POLICY_LIST);
        }
        case 'LAST_PROVISION_AT': {
            return (conditionKind = AlertCondition_1.ConditionKind.TIME_RANGE_TO_PAST);
        }
        case 'LAST_DEPROVISION_AT': {
            return (conditionKind = AlertCondition_1.ConditionKind.TIME_RANGE_TO_PAST);
        }
        case 'EXTENDED_MANAGEMENT': {
            return (conditionKind = AlertCondition_1.ConditionKind.BOOLEAN);
        }
        case 'ALIVE_AT': {
            return (conditionKind = AlertCondition_1.ConditionKind.TIME_RANGE_TO_PAST);
        }
        case 'NETWORK_INTERFACES': {
            return (conditionKind = AlertCondition_1.ConditionKind.WIFI_OR_ETHERNET);
        }
        case 'BATTERY_STATUS': {
            return (conditionKind = AlertCondition_1.ConditionKind.PERCENTAGE_RANGE);
        }
        case 'STORAGE_STATUS': {
            return (conditionKind = AlertCondition_1.ConditionKind.PERCENTAGE_RANGE);
        }
        case 'CURRENT_TIMEZONE': {
            return (conditionKind = AlertCondition_1.ConditionKind.MATCH_STRING);
        }
        case 'DISPLAY_SETTING_BACKLIGHT': {
            return (conditionKind = AlertCondition_1.ConditionKind.PERCENTAGE_RANGE);
        }
        case 'DISPLAY_SETTING_CONTRAST': {
            return (conditionKind = AlertCondition_1.ConditionKind.PERCENTAGE_RANGE);
        }
        case 'DISPLAY_SETTING_SHARPNESS': {
            return (conditionKind = AlertCondition_1.ConditionKind.PERCENTAGE_RANGE);
        }
        case 'DISPLAY_SETTING_MAX_TEMPERATURE': {
            return (conditionKind = AlertCondition_1.ConditionKind.NUMERIC_RANGE);
        }
        case 'INPUT_SOURCE': {
            return (conditionKind = AlertCondition_1.ConditionKind.INPUT_SOURCE);
        }
        case 'VOLUME': {
            return (conditionKind = AlertCondition_1.ConditionKind.PERCENTAGE_RANGE);
        }
        case 'BRIGHTNESS': {
            return (conditionKind = AlertCondition_1.ConditionKind.PERCENTAGE_RANGE);
        }
        case 'TIMERS': {
            return (conditionKind = AlertCondition_1.ConditionKind.TIMERS);
        }
        case 'PROPRIETARY_TIMERS': {
            return (conditionKind = AlertCondition_1.ConditionKind.PROPRIETARY_TIMERS);
        }
        case 'RESOLUTION': {
            return (conditionKind = AlertCondition_1.ConditionKind.RESOLUTION);
        }
        case 'ORIENTATION': {
            return (conditionKind = AlertCondition_1.ConditionKind.ORIENTATION);
        }
        case 'REMOTE_CONTROL': {
            return (conditionKind = AlertCondition_1.ConditionKind.BOOLEAN);
        }
        case 'APPLICATION_VERSION': {
            return (conditionKind = AlertCondition_1.ConditionKind.MATCH_SEMVER);
        }
        case 'FIRMWARE_VERSION': {
            return (conditionKind = AlertCondition_1.ConditionKind.MATCH_STRING);
        }
        case 'DEBUG': {
            return (conditionKind = AlertCondition_1.ConditionKind.BOOLEAN);
        }
        case 'POWER_ACTIONS_SCHEDULE': {
            return (conditionKind = AlertCondition_1.ConditionKind.SCHEDULED_POWER_ACTIONS);
        }
        case 'TEMPERATURE': {
            return (conditionKind = AlertCondition_1.ConditionKind.NUMERIC_RANGE);
        }
        case 'INSTALLED_PACKAGE': {
            return (conditionKind = AlertCondition_1.ConditionKind.MATCH_STRING);
        }
        case 'SCREENSHOT': {
            return (conditionKind = AlertCondition_1.ConditionKind.OCCURRENCES_IN_TIME_RANGE_TO_PAST);
        }
        case 'FEATURE_TESTS': {
            return (conditionKind = AlertCondition_1.ConditionKind.BOOLEAN);
        }
        case 'POLICY_VIOLATION_VOLUME': {
            return (conditionKind = AlertCondition_1.ConditionKind.BOOLEAN);
        }
        case 'POLICY_VIOLATION_BRIGHTNESS': {
            return (conditionKind = AlertCondition_1.ConditionKind.BOOLEAN);
        }
        case 'POLICY_VIOLATION_TIMERS': {
            return (conditionKind = AlertCondition_1.ConditionKind.BOOLEAN);
        }
        case 'POLICY_VIOLATION_PROPRIETARY_TIMERS': {
            return (conditionKind = AlertCondition_1.ConditionKind.BOOLEAN);
        }
        case 'POLICY_VIOLATION_RESOLUTION': {
            return (conditionKind = AlertCondition_1.ConditionKind.BOOLEAN);
        }
        case 'POLICY_VIOLATION_ORIENTATION': {
            return (conditionKind = AlertCondition_1.ConditionKind.BOOLEAN);
        }
        case 'POLICY_VIOLATION_REMOTE_CONTROL': {
            return (conditionKind = AlertCondition_1.ConditionKind.BOOLEAN);
        }
        case 'POLICY_VIOLATION_APPLICATION_VERSION': {
            return (conditionKind = AlertCondition_1.ConditionKind.BOOLEAN);
        }
        case 'POLICY_VIOLATION_FIRMWARE_VERSION': {
            return (conditionKind = AlertCondition_1.ConditionKind.BOOLEAN);
        }
        case 'POLICY_VIOLATION_DEBUG': {
            return (conditionKind = AlertCondition_1.ConditionKind.BOOLEAN);
        }
        case 'POLICY_VIOLATION_DATETIME': {
            return (conditionKind = AlertCondition_1.ConditionKind.BOOLEAN);
        }
        case 'POLICY_VIOLATION_POWER_ACTIONS_SCHEDULE': {
            return (conditionKind = AlertCondition_1.ConditionKind.BOOLEAN);
        }
        case 'POLICY_VIOLATION_TEMPERATURE': {
            return (conditionKind = AlertCondition_1.ConditionKind.BOOLEAN);
        }
        case 'APPLET_COMMAND': {
            return (conditionKind = AlertCondition_1.ConditionKind.STRING_OCCURRENCES_IN_TIME_RANGE_TO_PAST);
        }
    }
    return conditionKind;
}
class InvalidBooleanCondition extends Error {
    constructor(condition) {
        super(`Condition is not valid boolean: ${JSON.stringify(condition)}`);
    }
}
exports.InvalidBooleanCondition = InvalidBooleanCondition;
class InvalidOccurrencesInTimeRangeCondition extends Error {
    constructor(condition) {
        super(`Condition is not valid occurrences in time range: ${JSON.stringify(condition)}`);
    }
}
exports.InvalidOccurrencesInTimeRangeCondition = InvalidOccurrencesInTimeRangeCondition;
class InvalidStringOccurencesInTimeRangeCondition extends Error {
    constructor(condition) {
        super(`Condition is not valid string occurrences in time range: ${JSON.stringify(condition)}`);
    }
}
exports.InvalidStringOccurencesInTimeRangeCondition = InvalidStringOccurencesInTimeRangeCondition;
class InvalidNumericRangeCondition extends Error {
    constructor(condition) {
        super(`Condition is not valid operator and range: ${JSON.stringify(condition)}`);
    }
}
exports.InvalidNumericRangeCondition = InvalidNumericRangeCondition;
class InvalidCondition extends Error {
    constructor(condition) {
        super(`Condition is not valid: ${JSON.stringify(condition)}`);
    }
}
exports.InvalidCondition = InvalidCondition;
class InvalidConditionToAlertType extends Error {
    constructor(condition, alertType) {
        super(`Conditions : ${JSON.stringify(condition)} are not valid for alert type : ${JSON.stringify(alertType)}`);
    }
}
exports.InvalidConditionToAlertType = InvalidConditionToAlertType;
