export interface ClientOptions {
    /** Callback that is called when the device is connected or disconnected */
    onStatus?: (connected: boolean) => void;
    /**
     * Timeout in milliseconds for establishing a connection between device and browser client.
     * When the timeout is reached, the promise `StartedClient.onceConnected()` is rejected.
     * If the connection establishment times out, the client is stopped.
     *
     * Default timeout is disabled.
     */
    timeoutMs?: number;
}
export interface Client {
    /**
     * Starts the client and connects to the server.
     * Promise is resolved when the remote desktop server accepts the connection
     * and the authentication is successful.
     *
     * For checking of establishing a connection between device and browser client,
     * use `onceConnected()` method on returned StartedClient or option `onStatus`.
     */
    start(options: ClientOptions): Promise<StartedClient>;
    /**
     * Stops the client and disconnects from the server.
     */
    stop(): Promise<void>;
    /**
     * Returns true if the client is started and connected to the server.
     * Started doesn't mean that the connection between device and browser client
     * is established.
     */
    isStarted(): boolean;
}
export interface StartedClient {
    /**
     * Stops the client and disconnects from the server.
     * It's an alias for `Client.stop()` method.
     */
    stop(): Promise<void>;
    /**
     * Returns a promise that resolves when the operator
     * established a connection between device and browser client.
     *
     * It's derived from `onStatus(connected = true)` event.
     *
     * Alternatively, the promise can be rejected if the connection
     * failed, the client is closed before the connection is established,
     * or the connection timed out.
     */
    onceConnected(): Promise<void>;
}
export declare function createStartedClient(client: Client, connectedPromise: Promise<void>, options: ClientOptions): StartedClient;
