# Remote Desktop
Sharing a screen of remote devices to the browser and controlling them using a mouse and keyboard.

### Docker-compose

There is also prepared `docker-compose.yml` ready for your development. It starts the services.
Be sure you copy `.env.docker-compose.dist` to `.env.docker-compose` and fill your own valid token. Then you may run the tests by:
```
docker-compose up -d
# make sure there are correct dependencies
docker-compose exec app rm -rf node_modules
# you must login to sOS npm before installing private packages
docker-compose exec app npm login --registry=https://npm.signageos.io/
docker-compose exec app npm install

docker-compose exec app npm run lint
docker-compose exec app npm run test
```

## Client Usage
### Browser Client
```ts
import { BrowserClient } from '@signageos/remote-desktop';

const screenshotRenderer = {
	render({ screenshotUri, width, height }) {
		console.log('rendering screenshot', { screenshotUri, width, height });
	},
};
const browserClient = new BrowserClient('device-uid', screenshotRenderer);
const startedBrowserClient = await browserClient.start({
	serverUrl: 'https://remote-desktop.signageos.io',
	auth: { tokenId: 'token-id', token: 'token' },
});
await startedBrowserClient.onceConnected();
// ...now you can start sending mouse events
await startedBrowserClient.stop();
```

## Architecture
This component is a Server part for 2 client components that are connected over the same device (identified by DUID or Identity Hash).

### Components
- **Browser Client** - Browser client is a client that is connected to the remote-desktop server and sends mouse events to the server and receives screenshots from the server.
- **Device Client** - Device client is a client that is connected to the remote-desktop server and receives mouse events from the server and sends screenshots to the server.

```mermaid
graph
	browser[Browser Client] -->|Sends mouse events| remote-desktop[Remote Desktop Server]
	remote-desktop -->|Forwards mouse events| device[Device Client]
	device -->|Sends screenshots| remote-desktop
	remote-desktop -->|Forwards screenshots| browser
	remote-desktop -->|Authenticates| platform-db[Platform DB]
	user(User) -->|Interacts| browser
```

### Authentication
Device Client is only identified by DUID of the device.

Browser Client is identified by Identity Hash of the user. And additionally it is authenticated using `x-auth` query param with valid tokenId and token concatenated by `:` character.

### Flow Chart
```mermaid
sequenceDiagram
	participant Device
	participant Browser
	participant RemoteDesktop
	participant PlatformDB

	Device->>RemoteDesktop: Connect with `duid` query param
	RemoteDesktop->>PlatformDB: Identify Device
	PlatformDB-->>RemoteDesktop: Device Identity
	Note right of Device: Waiting for connection from Browser

	Browser->>RemoteDesktop: Connect with `deviceUid` query param and `x-auth` query param
	RemoteDesktop->>PlatformDB: Authenticate Browser and Organization
	PlatformDB-->>RemoteDesktop: Browser Identity
	Note right of Browser: Both sides are connected, trigger connected events

	RemoteDesktop->>Device: Send `BrowserConnected` event
	RemoteDesktop->>Browser: Send `DeviceConnected` event

	Note right of Device: Starts sending screenshots stream
	loop ScreenStream
		Device->>RemoteDesktop: Send `ScreenshotRecorded` event
		RemoteDesktop->>Browser: Send `ScreenshotRecorded` event
	end

	Note right of Browser: User interacts by sending mouse events
	loop MouseEvents
		Browser->>RemoteDesktop: Send `MouseEvent` event
		RemoteDesktop->>Device: Send `MouseEvent` event
	end
```
