"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.formatIfZodError = exports.formatZodError = exports.FormattedZodError = exports.formatZodErrorMessage = exports.isExternalZodError = exports.isZodError = void 0;
const zod_1 = require("zod");
/**
 * Detect whether the object is a ZodError (instance of ZodError could not work)
 */
function isZodError(error) {
    if (!error || typeof error !== 'object' || !('name' in error)) {
        return false;
    }
    return error instanceof zod_1.ZodError || (error.name === 'ZodError' && 'issues' in error);
}
exports.isZodError = isZodError;
/**
 * Detect whether the object is an external error with Zod-like originalError inside
 */
function isExternalZodError(error) {
    if (!error || typeof error !== 'object') {
        return false;
    }
    return 'originalError' in error && typeof error.originalError === 'object' && isZodError(error.originalError);
}
exports.isExternalZodError = isExternalZodError;
const formatZodErrorMessage = (error) => {
    return error.issues.map((issue) => `${issue.path.length > 0 ? issue.path + ': ' : ''}${issue.message}`).join('\n');
};
exports.formatZodErrorMessage = formatZodErrorMessage;
class FormattedZodError extends Error {
    constructor(message, zodError) {
        super(message);
        this.zodError = zodError;
        Object.setPrototypeOf(this, FormattedZodError.prototype);
    }
}
exports.FormattedZodError = FormattedZodError;
/**
 * Returns the alternative instance of FormettedZodError
 * that contains the original zodError but the message
 * is formatted as a human readable single string.
 */
const formatZodError = (error) => {
    const message = (0, exports.formatZodErrorMessage)(error);
    return new FormattedZodError(message, error);
};
exports.formatZodError = formatZodError;
/**
 * Format only if the error is ZodError
 * @see formatZodError
 */
const formatIfZodError = (error) => {
    if (isZodError(error)) {
        return (0, exports.formatZodError)(error);
    }
    return error;
};
exports.formatIfZodError = formatIfZodError;
//# sourceMappingURL=zodError.js.map