import { DBConnections } from './dbConnections';
import { ServiceConfig } from '../Config/config';
import { IsReadyCallback } from './HTTP/routes/monitoring/readniness';
import { IsAliveCallback } from './HTTP/routes/monitoring/aliveness';
import { IService } from './types';
import { OnStopCallback } from './Service';
import { AccessLogCallback } from './HTTP/express/app';
/**
 * Create instance of a service
 *
 * Service implements all functionality that is common for all services
 * This functionality is the same in every service.
 *
 * We had a problem of writing it from scratch each time for every service.
 * Sometimes there was a drift, where some new ideas were implemented in one or more services
 * but not in all services.
 *
 * This single common implementation aims to prevent that and provide a common implementation
 * that all services can use as a library.
 *
 * @param config - It's recommended to use the direct output of `parseServiceConfig()`. The format is deliberately the same.
 * @param connections - Database connections
 * @param isReady - Custom readiness check for Kubernetes
 * @param isAlive - Custom aliveness check for Kubernetes
 * @param accessLogCallback - Custom callback to modify the access log before it is logged
 *
 * -------
 *
 * Adding application-specific HTTP routes:
 *
 * This factory returns express app, that can be used to add more routes.
 *
 * For example:
 *
 *   ```typescript
 *   const service = createService({ ... });
 *   service.httpServer.express.get('/my-route', (req, res) => {
 *     res.send('Hello World!');
 *   });
 *   ```
 *
 * -------
 *
 * Adding application-specific metrics:
 *
 * By default, the service serves some basic Node.js metrics.
 * This factory returns a metrics client, that can be used to add more metrics.
 *
 * For example:
 *
 *  ```typescript
 *  const service = createService({ ... });
 *  const client = service.metricsClient.getClient();
 *  const myCounter = new client.Counter({
 *    name: 'my_counter',
 *    help: 'This is my counter',
 *  });
 *
 *  // now you can independently increment the counter in your code
 *  // and the value will automatically be exposed when serving metrics
 *  ```
 *
 *  -------
 *
 *  Custom readiness and aliveness checks:
 *
 * By default, the service serves `/ready` and `/alive` routes, that are used by Kubernetes readiness/aliveness probes.
 * You can provide custom callbacks if the default checks are not enough:
 *
 *  ```typescript
 *  const service = createService({
 *    // ...
 *
 *    isReady: async () => {
 * 			// your custom readiness check
 * 			// either return true/false if service is/isn't ready to receive traffic
 * 			// or return an object { ready: boolean, response: any } to provide a custom response
 * 			return true;
 * 		},
 *
 *    isAlive: async () => {
 * 			// your custom aliveness check
 * 			// return true/false if service is/isn't alive
 * 			return true;
 * 		},
 *  });
 *  ```
 */
export declare function createService({ config, connections, isReady, isAlive, accessLogCallback, onStopCallback, }: {
    config: Pick<ServiceConfig, 'app' | 'sentry'>;
    connections?: DBConnections;
    isReady?: IsReadyCallback;
    isAlive?: IsAliveCallback;
    /**
     * Custom callback to modify the access log before it is logged
     *
     * Use it to add custom fields to the access log.
     * It's not possible to modify the access log directly, only to add additional fields.
     *
     * @returns metadata object that will be merged with the access log
     */
    accessLogCallback?: AccessLogCallback;
    /**
     * Custom callback to run when the service is stopped
     *
     * This callback is called when the service receives a stop signal or when the service method stop() is called.
     *
     * This callback should perform any necessary cleanup before the service and the entire process is stopped.
     * Typically that would include:
     *   - draining connections
     *   - waiting for all pending requests to finish
     *   - calling external services to notify them about the shutdown
     *
     * @note database connections are automatically closed when the service is stopped.
     */
    onStopCallback?: OnStopCallback;
}): IService;
