"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.bindFailed = exports.bindUnspecificFromDevice = exports.bindSpecificFromDevice = exports.bindToDevice = exports.sendFromDevice = exports.sendToDevice = void 0;
const errors_1 = require("../errors");
const EXCHANGE_ACTIONS_TO_DEVICE = 'actionsToDevice';
const EXCHANGE_ACTIONS_NOT_DELIVERED_TO_DEVICE = 'actionsNotDeliveredToDevice';
const EXCHANGE_ACTIONS_FROM_DEVICE = 'actionsFromDevice';
const ALTERNATE_EXCHANGE_ACTIONS_FROM_DEVICE = 'alternateActionsFromDevice';
const UNSPECIFIC_ACTIONS_FROM_DEVICE_QUEUE_NAME = 'unspecificActionsFromDevice';
const FAILED_ACTIONS_CONSUMER_QUEUE_NAME = 'actionsNotDeliveredToDevice';
const DEFAULT_QUEUE_OPTIONS = {
    persistent: false,
    confirmable: true,
    prefetchCount: 500,
};
const TO_DEVICE_CONSUME_OPTIONS = {
    // to device queue is exclusive
    autoDelete: false,
    // to device queue is not recovered on restart
    durable: false,
    // to device queue is exclusive
    exclusive: true,
    respond: true,
};
const FROM_DEVICE_CONSUME_OPTIONS = {
    // queue from device and not delivered are persistent (not autoDelete)
    // will be kept even if 0 consumers (could happen only during BC deploys)
    autoDelete: false,
    // queue is durable (will not be deleted on restart to keep messages to be processed later)
    durable: true,
    queueType: 'quorum',
};
/**
 * @param postponable Postponable actions has to be delivered. When device is not connected at the time, it's postponed for later.
 * @param sessionUid Select a specific session (socket) to send message to. If specified, no other socket of single device get the action.
 * @param sendTimeout Specify how long it will wait for response until action is enqueued as not delivered action.
 */
function sendToDevice(amqpConnection, action, deviceUid, postponable, sessionUid, sendTimeout) {
    return __awaiter(this, void 0, void 0, function* () {
        const headers = {
            deviceUid,
            sessionUid,
            actionType: action.type,
            postponable,
        };
        try {
            const enqueueArgs = [
                createChannelToDevice(amqpConnection, undefined),
                action,
                headers,
                EXCHANGE_ACTIONS_TO_DEVICE,
                action.type,
                {
                    // delivery to device is not persistent message
                    persistent: false,
                    // expire messages after some time to prevent queue growing large in case of consumer failure
                    expiration: sendTimeout * 2,
                },
            ];
            const expectingResponse = !Boolean(sessionUid);
            if (expectingResponse) {
                yield amqpConnection.queuePublisher.enqueueExpectingResponse(...enqueueArgs, sendTimeout);
            }
            else {
                yield amqpConnection.queuePublisher.enqueue(...enqueueArgs);
            }
        }
        catch (error) {
            if (error instanceof errors_1.ResponseTimeoutError) {
                yield enqueueFailedAction(amqpConnection, action, deviceUid, postponable, sessionUid);
            }
            else {
                throw error;
            }
        }
    });
}
exports.sendToDevice = sendToDevice;
function enqueueFailedAction(amqpConnection, action, deviceUid, postponable, sessionUid) {
    return __awaiter(this, void 0, void 0, function* () {
        const headers = {
            deviceUid,
            sessionUid,
            actionType: action.type,
            postponable,
        };
        yield amqpConnection.queuePublisher.enqueue(createChannelToDevice(amqpConnection, undefined), action, headers, EXCHANGE_ACTIONS_NOT_DELIVERED_TO_DEVICE, '', // fanout
        { persistent: true });
    });
}
/**
 * @param timestamp of action when it was sent from the device
 */
function sendFromDevice(amqpConnection, action, deviceUid, sessionUid, socketName, timestamp) {
    return __awaiter(this, void 0, void 0, function* () {
        const headers = {
            deviceUid,
            sessionUid,
            socketName,
            timestamp,
        };
        yield amqpConnection.queuePublisher.enqueue(createChannelFromDevice(amqpConnection), action, headers, EXCHANGE_ACTIONS_FROM_DEVICE, // all actions are going through one queue from device
        action.type, // routingKey=actionType, messages are routed to specific queue or to unspecific if nothing bound
        { persistent: true });
    });
}
exports.sendFromDevice = sendFromDevice;
function bindToDevice(amqpConnection, consumerName, processUid, onAction) {
    return __awaiter(this, void 0, void 0, function* () {
        const onRelatedAction = (action, headers, ack, nack) => __awaiter(this, void 0, void 0, function* () {
            yield onAction(headers.deviceUid, headers.sessionUid, action, ack, nack);
        });
        const cancelConsumption = yield amqpConnection.queueSubscriber.subscribeExpectingConfirmationRepeatable(createChannelToDevice(amqpConnection, processUid), getActionsToDeviceQueueName(consumerName, processUid), [{ exchangeName: EXCHANGE_ACTIONS_TO_DEVICE, persistent: false }], onRelatedAction, TO_DEVICE_CONSUME_OPTIONS);
        return cancelConsumption;
    });
}
exports.bindToDevice = bindToDevice;
/**
 * Binds action messages by a specific actionTypes to a single queue. The queue is named after the input group.
 * So, more action types are grouped together into single queue. This is preventing from stucking messages across more
 * priority leveled groups.
 * E.g.: The priority actions used for authentication, registration & verification should not be affected by telemetry actions in any way.
 * The priority actions has much more strict requirements on SLA.
 *
 * @param group the string name of group which will be used as a part of queue name.
 * @param actionTypes all types of actions which are bound to the queue.
 * @param onAction the callback which is called for every action bound above.
 */
function bindSpecificFromDevice(amqpConnection, group, actionTypes, onAction, options) {
    return __awaiter(this, void 0, void 0, function* () {
        const fullOptions = Object.assign(Object.assign({}, FROM_DEVICE_CONSUME_OPTIONS), options);
        return yield amqpConnection.queueSubscriber.subscribeRepeatable(createChannelSpecificFromDevice(amqpConnection, group), getSpecificActionsFromDeviceQueueName(group), actionTypes.map((actionType) => ({ exchangeName: EXCHANGE_ACTIONS_FROM_DEVICE, persistent: true, routingKey: actionType })), (action, headers) => onAction(action, headers.deviceUid, headers.sessionUid, headers.socketName, headers.timestamp), fullOptions);
    });
}
exports.bindSpecificFromDevice = bindSpecificFromDevice;
/**
 * According to bindSpecificFromDevice, for all action types which were not bound there, the actions are fall backed to this consumption.
 * Those unspecific actions has self alternate exchange and appropriate single queue.
 * This consumption can be used for all not specifically bound actions, for example action logs of devices.
 *
 * @param onAction the callback which is called for every action ever not specifically bound by bindSpecificFromDevice.
 */
function bindUnspecificFromDevice(amqpConnection, onAction, options) {
    return __awaiter(this, void 0, void 0, function* () {
        const fullOptions = Object.assign(Object.assign({}, FROM_DEVICE_CONSUME_OPTIONS), options);
        return yield amqpConnection.queueSubscriber.subscribeRepeatable(createChannelUnspecificFromDevice(amqpConnection), UNSPECIFIC_ACTIONS_FROM_DEVICE_QUEUE_NAME, [{ exchangeName: ALTERNATE_EXCHANGE_ACTIONS_FROM_DEVICE, persistent: true }], (action, headers) => onAction(action, headers.deviceUid, headers.sessionUid, headers.socketName, headers.timestamp), fullOptions);
    });
}
exports.bindUnspecificFromDevice = bindUnspecificFromDevice;
function bindFailed(amqpConnection, onAction, options) {
    return __awaiter(this, void 0, void 0, function* () {
        const fullOptions = Object.assign(Object.assign({}, FROM_DEVICE_CONSUME_OPTIONS), options);
        const onActionCallback = (action, headers, ack, nack) => __awaiter(this, void 0, void 0, function* () {
            yield onAction(action, headers.deviceUid, headers.postponable, ack, nack);
        });
        return yield amqpConnection.queueSubscriber.subscribeExpectingConfirmationRepeatable(createChannelFailedToDevice(amqpConnection), FAILED_ACTIONS_CONSUMER_QUEUE_NAME, // failed messages are flat in one queue
        [
            // bind all not delivered actions (dead-lettered during delivering to device)
            { exchangeName: EXCHANGE_ACTIONS_NOT_DELIVERED_TO_DEVICE, persistent: true },
        ], onActionCallback, fullOptions);
    });
}
exports.bindFailed = bindFailed;
function createChannelFailedToDevice(amqpConnection, options) {
    const FAILED_ACTIONS_CHANNEL_NAME = 'actionsNotDeliveredToDevice';
    const fullOptions = Object.assign(Object.assign({}, DEFAULT_QUEUE_OPTIONS), options);
    return () => __awaiter(this, void 0, void 0, function* () {
        const channel = yield amqpConnection.channelProvider.getChannel(FAILED_ACTIONS_CHANNEL_NAME, fullOptions);
        yield assertExchangeActionsNotDeliveredToDevice(channel);
        return channel;
    });
}
function createChannelFromDevice(amqpConnection, options) {
    const FROM_DEVICE_CHANNEL_NAME = 'actionsFromDevice';
    const fullOptions = Object.assign(Object.assign({}, DEFAULT_QUEUE_OPTIONS), options);
    return () => __awaiter(this, void 0, void 0, function* () {
        const channel = yield amqpConnection.channelProvider.getChannel(FROM_DEVICE_CHANNEL_NAME, fullOptions);
        yield assertExchangeActionsFromDevice(channel);
        return channel;
    });
}
function createChannelSpecificFromDevice(amqpConnection, group, options) {
    const fromDeviceChannelName = `specificActionsFromDevice.${group}`;
    const fullOptions = Object.assign(Object.assign({}, DEFAULT_QUEUE_OPTIONS), options);
    return () => __awaiter(this, void 0, void 0, function* () {
        const channel = yield amqpConnection.channelProvider.getChannel(fromDeviceChannelName, fullOptions);
        yield assertExchangeActionsFromDevice(channel);
        return channel;
    });
}
function createChannelUnspecificFromDevice(amqpConnection, options) {
    const FROM_DEVICE_CHANNEL_NAME = 'unspecificActionsFromDevice';
    const fullOptions = Object.assign(Object.assign({}, DEFAULT_QUEUE_OPTIONS), options);
    return () => __awaiter(this, void 0, void 0, function* () {
        const channel = yield amqpConnection.channelProvider.getChannel(FROM_DEVICE_CHANNEL_NAME, fullOptions);
        yield assertAlternateExchangeActionsFromDevice(channel);
        return channel;
    });
}
function createChannelToDevice(amqpConnection, processUid, options) {
    const TO_DEVICE_CHANNEL_NAME = 'actionsToDevice' + (typeof processUid === 'undefined' ? '' : '.' + processUid);
    const fullOptions = Object.assign(Object.assign({}, DEFAULT_QUEUE_OPTIONS), options);
    return () => __awaiter(this, void 0, void 0, function* () {
        const channel = yield amqpConnection.channelProvider.getChannel(TO_DEVICE_CHANNEL_NAME, fullOptions);
        yield assertExchangeActionsToDevice(channel);
        return channel;
    });
}
function assertExchangeActionsToDevice(channel) {
    return __awaiter(this, void 0, void 0, function* () {
        yield channel.assertExchange(EXCHANGE_ACTIONS_TO_DEVICE, 'fanout');
    });
}
function assertExchangeActionsNotDeliveredToDevice(channel) {
    return __awaiter(this, void 0, void 0, function* () {
        yield channel.assertExchange(EXCHANGE_ACTIONS_NOT_DELIVERED_TO_DEVICE, 'fanout');
    });
}
function assertExchangeActionsFromDevice(channel) {
    return __awaiter(this, void 0, void 0, function* () {
        yield channel.assertExchange(EXCHANGE_ACTIONS_FROM_DEVICE, 'direct', {
            alternateExchange: ALTERNATE_EXCHANGE_ACTIONS_FROM_DEVICE,
        });
    });
}
function assertAlternateExchangeActionsFromDevice(channel) {
    return __awaiter(this, void 0, void 0, function* () {
        yield channel.assertExchange(ALTERNATE_EXCHANGE_ACTIONS_FROM_DEVICE, 'fanout');
    });
}
function getActionsToDeviceQueueName(consumerName, processUid) {
    return 'actionsToDevice.' + consumerName + '.' + processUid;
}
function getSpecificActionsFromDeviceQueueName(group) {
    return 'specificActionsFromDevice.' + group;
}
//# sourceMappingURL=deviceActionsQueue.js.map