"use strict";
var __assign = (this && this.__assign) || function () {
    __assign = Object.assign || function(t) {
        for (var s, i = 1, n = arguments.length; i < n; i++) {
            s = arguments[i];
            for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p))
                t[p] = s[p];
        }
        return t;
    };
    return __assign.apply(this, arguments);
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.MongoDBDriverExporter = void 0;
var prom_client_1 = require("prom-client");
var debug_1 = __importDefault(require("debug"));
var debug = (0, debug_1.default)('@signageos/lib:Metrics:mongoDBDriverExporter');
var defaultOptions = {
    mongodbDriverCommandsSecondsHistogramBuckets: [0.001, 0.005, 0.01, 0.02, 0.03, 0.04, 0.05, 0.1, 0.2, 0.5, 1, 2, 5, 10],
};
/**
 * Exports metrics for the MongoDB driver
 *
 * It has to be a singleton because it registers the metrics in the global registry
 * and it's not possible to register the same metric multiple times.
 */
var MongoDBDriverExporter = /** @class */ (function () {
    function MongoDBDriverExporter(register, options) {
        this.register = register;
        this.options = __assign(__assign({}, defaultOptions), options);
        var connectionPoolLabels = ['name'];
        this.poolSize = new prom_client_1.Gauge({
            name: 'mongodb_driver_pool_size',
            help: 'the current size of the connection pool, including idle and in-use members',
            labelNames: connectionPoolLabels,
            registers: [this.register],
        });
        this.minSize = new prom_client_1.Gauge({
            name: 'mongodb_driver_pool_min',
            help: 'the minimum size of the connection pool',
            labelNames: connectionPoolLabels,
            registers: [this.register],
        });
        this.maxSize = new prom_client_1.Gauge({
            name: 'mongodb_driver_pool_max',
            help: 'the maximum size of the connection pool',
            labelNames: connectionPoolLabels,
            registers: [this.register],
        });
        this.active = new prom_client_1.Gauge({
            name: 'mongodb_driver_pool_active',
            help: 'the count of connections that are currently in use',
            labelNames: connectionPoolLabels,
            registers: [this.register],
        });
        this.wait = new prom_client_1.Gauge({
            name: 'mongodb_driver_pool_wait',
            help: 'the current size of the wait queue for a connection from the pool',
            labelNames: connectionPoolLabels,
            registers: [this.register],
        });
        this.commandsHistogram = new prom_client_1.Histogram({
            name: 'mongodb_driver_commands_seconds',
            help: 'Timer of mongodb commands',
            buckets: this.options.mongodbDriverCommandsSecondsHistogramBuckets,
            labelNames: ['name', 'command'],
            registers: [this.register],
        });
        this.failedCommands = new prom_client_1.Counter({
            name: 'mongodb_driver_commands_failed_total',
            help: 'Counter of failed mongodb commands',
            labelNames: ['name'],
            registers: [this.register],
        });
    }
    /**
     * Start monitoring the given mongo client instance
     *
     * @param mongoClient mongo client instance
     * @param name unique name for the client that will be used as a label to differentiate between multiple clients
     */
    MongoDBDriverExporter.prototype.monitorMongoClient = function (mongoClient, name) {
        var _this = this;
        var labels = { name: name };
        mongoClient.on('connectionPoolCreated', function (event) { return _this.onConnectionPoolCreated(labels, event); });
        mongoClient.on('connectionPoolClosed', function () { return _this.onConnectionPoolClosed(labels); });
        mongoClient.on('connectionCreated', function () { return _this.onConnectionCreated(labels); });
        mongoClient.on('connectionClosed', function () { return _this.onConnectionClosed(labels); });
        mongoClient.on('connectionCheckOutStarted', function () { return _this.onConnectionCheckOutStarted(labels); });
        mongoClient.on('connectionCheckedOut', function () { return _this.onConnectionCheckedOut(labels); });
        mongoClient.on('connectionCheckOutFailed', function () { return _this.onConnectionCheckOutFailed(labels); });
        mongoClient.on('connectionCheckedIn', function () { return _this.onConnectionCheckedIn(labels); });
        mongoClient.on('commandSucceeded', function (event) { return _this.onCommandSucceeded(labels, event); });
        mongoClient.on('commandFailed', function (event) { return _this.onCommandFailed(labels, event); });
    };
    MongoDBDriverExporter.prototype.onConnectionPoolCreated = function (labels, event) {
        debug('connectionPoolCreated', event);
        this.poolSize.set(labels, 0);
        this.minSize.set(labels, event.options.minPoolSize);
        this.maxSize.set(labels, event.options.maxPoolSize);
        this.active.set(labels, 0);
        this.wait.set(labels, 0);
    };
    MongoDBDriverExporter.prototype.onConnectionCreated = function (labels) {
        debug('connectionCreated');
        this.poolSize.inc(labels);
    };
    MongoDBDriverExporter.prototype.onConnectionClosed = function (labels) {
        debug('connectionClosed');
        this.poolSize.dec(labels);
    };
    MongoDBDriverExporter.prototype.onConnectionCheckOutStarted = function (labels) {
        debug('connectionCheckOutStarted');
        this.wait.inc(labels);
    };
    MongoDBDriverExporter.prototype.onConnectionCheckedOut = function (labels) {
        debug('connectionCheckedOut');
        this.active.inc(labels);
        this.wait.dec(labels);
    };
    MongoDBDriverExporter.prototype.onConnectionCheckOutFailed = function (labels) {
        debug('connectionCheckOutFailed');
        this.wait.dec(labels);
    };
    MongoDBDriverExporter.prototype.onConnectionCheckedIn = function (labels) {
        debug('connectionCheckedIn');
        this.active.dec(labels);
    };
    MongoDBDriverExporter.prototype.onConnectionPoolClosed = function (labels) {
        debug('connectionPoolClosed');
        this.poolSize.set(labels, 0);
        this.minSize.set(labels, 0);
        this.maxSize.set(labels, 0);
        this.active.set(labels, 0);
        this.wait.set(labels, 0);
    };
    MongoDBDriverExporter.prototype.onCommandSucceeded = function (labels, event) {
        debug('commandSucceeded', event);
        var fullLabels = __assign(__assign({}, labels), { command: event.commandName });
        this.commandsHistogram.observe(fullLabels, event.duration / 1000);
    };
    MongoDBDriverExporter.prototype.onCommandFailed = function (labels, event) {
        debug('commandFailed', event);
        this.failedCommands.inc(labels);
    };
    return MongoDBDriverExporter;
}());
exports.MongoDBDriverExporter = MongoDBDriverExporter;
//# sourceMappingURL=mongoDBDriverExporter.js.map