import { S3ClientConfig } from '@aws-sdk/client-s3';
export interface UploadFileArgs {
    /** Path to local file */
    filePath: string;
    /** Destination bucket to upload the file to */
    bucket: string;
    /** Path in the bucket to upload the file to */
    remotePath: string;
    /**
     * A standard MIME type describing the format of the contents
     * @see https://www.rfc-editor.org/rfc/rfc9110.html#name-content-type
     */
    contentType?: string;
}
export interface UploadContentArgs {
    /** Content of the file to upload */
    content: string;
    /** Destination bucket to upload the file to */
    bucket: string;
    /** Path in the bucket to upload the file to */
    remotePath: string;
    /**
     * A standard MIME type describing the format of the contents
     * @see https://www.rfc-editor.org/rfc/rfc9110.html#name-content-type
     */
    contentType?: string;
}
export interface DownloadFileArgs {
    bucket: string;
    remotePath: string;
    filePath: string;
}
export interface ListFilesArgs {
    bucket: string;
    remotePath: string;
}
/**
 * Wrapper for AWS S3 client
 *
 * All commonly used operations with S3 will be implemented here to provide a consistent interface across the codebase.
 */
export interface IAwsS3 {
    /**
     * Upload a local file to S3
     */
    uploadFile(args: UploadFileArgs): Promise<void>;
    /**
     * Upload contents of a file stored in memory directly to S3
     */
    uploadContent(args: UploadContentArgs): Promise<void>;
    /**
     * Download a file from S3 to local file system
     */
    downloadFile({ bucket, remotePath, filePath }: DownloadFileArgs): Promise<void>;
    /**
     * List files in a particular path in a bucket
     */
    listFiles({ bucket, remotePath }: ListFilesArgs): Promise<string[]>;
}
/**
 * Wrapper for AWS S3 client
 *
 * All commonly used operations with S3 will be implemented here to provide a consistent interface across the codebase.
 */
export declare class AwsS3 implements IAwsS3 {
    private client;
    /**
     * @param config.forcePathStyle - Whether to use path style URLs for S3 requests.
     * Defaults to the value of the `AWS_S3_FORCE_PATH_STYLE_URLS` environment variable or false if not set.
     */
    constructor(config?: S3ClientConfig);
    uploadFile({ filePath, bucket, remotePath, contentType }: UploadFileArgs): Promise<void>;
    uploadContent({ content, bucket, remotePath, contentType }: UploadContentArgs): Promise<void>;
    downloadFile({ bucket, remotePath, filePath }: DownloadFileArgs): Promise<void>;
    listFiles({ bucket, remotePath }: ListFilesArgs): Promise<string[]>;
}
