import { ChangeLogEntry, isValidChangeLogEntry } from './ChangeLogEntry';
import { promises as fs } from 'fs';

const VERSION_HEADER = '## ';
const CHANGES_HEADER = '### ';
const LOG_LINE = '- ';

export const getPipedInput = function () {
	return new Promise<string>((resolve, reject) => {
		let inputData: string;
		process.stdin.resume();
		process.stdin.setEncoding('utf8');
		process.stdin.on('data', (chunk) => (inputData += chunk));
		process.stdin.on('end', () => resolve(inputData));
		process.stdin.on('error', (error) => reject(error));
	});
};

export const getGroupedChangeLog = function (changeLog: string) {
	const newLines = changeLog
		.split(/\n|\r/)
		.map((line) => line.trim())
		.filter((line) => line.startsWith(VERSION_HEADER) || line.startsWith(CHANGES_HEADER) || line.startsWith(LOG_LINE));
	const { groupedLog } = newLines.reduce(
		(reduction, line) => {
			if (line.startsWith(VERSION_HEADER)) {
				const version = line
					.substring(VERSION_HEADER.length)
					.match(/^\[?(([\d\w\.\-\+])+)\]?/)?.[1]
					?.trim();
				if (version === undefined) {
					throw new Error(`Invalid version format "${version}"`);
				}
				if (reduction.groupedLog[version] !== undefined) {
					throw new Error(`The version "${version}" is duplicated in CHANGELOG`);
				}
				return {
					...reduction,
					lastVersion: version,
					groupedLog: {
						...reduction.groupedLog,
						[version]: {},
					},
				};
			} else if (line.startsWith(CHANGES_HEADER)) {
				const type = line.substring(CHANGES_HEADER.length).trim();
				const versionGroup = reduction.groupedLog[reduction.lastVersion!];
				if (versionGroup === undefined || reduction.lastVersion === null) {
					throw new Error(`The type "${type}" is not under any version section in CHANGELOG`);
				}
				if (!isValidChangeLogEntry(type)) {
					throw new Error(`Invalid type entry ${type}`);
				}
				if (versionGroup[type] !== undefined) {
					throw new Error(`The type "${type}" is duplicated at version "${reduction.lastVersion}" in CHANGELOG`);
				}
				return {
					...reduction,
					lastType: type,
					groupedLog: {
						...reduction.groupedLog,
						[reduction.lastVersion]: {
							...reduction.groupedLog[reduction.lastVersion],
							[type]: versionGroup[type] || [],
						},
					},
				};
			} else {
				if (reduction.lastVersion === null) {
					throw new Error(`Line "${line}" is not under any version section in CHANGELOG`);
				}
				if (reduction.lastType === null) {
					throw new Error(`Line "${line}" is not under any type section in CHANGELOG`);
				}

				return {
					...reduction,
					groupedLog: {
						...reduction.groupedLog,
						[reduction.lastVersion]: {
							...reduction.groupedLog[reduction.lastVersion],
							[reduction.lastType]: [
								...reduction.groupedLog[reduction.lastVersion]![reduction.lastType!]!,
								line.substring(LOG_LINE.length),
							],
						},
					},
				};
			}
		},
		{ groupedLog: {}, lastType: null, lastVersion: null } as {
			groupedLog: Record<string, Partial<Record<ChangeLogEntry, string[] | undefined>>>;
			lastType: ChangeLogEntry | null;
			lastVersion: string | null;
		},
	);
	return groupedLog;
};

export const readJSON = (path: string) => fs.readFile(path).then((b) => JSON.parse(b.toString()));

export const detectIndent = async (path: string) => (/^\t/.test(await fs.readFile(path).then((b) => b.toString())) ? '\t' : '  ');

export interface PackageJson {
	name: string;
	version: string;

	dependencies: Record<string, string>;
	devDependencies: Record<string, string>;
}

export interface PackageLockJson {
	version: string;
	dependencies: Record<
		string,
		{
			version: string;
		}
	>;
	packages: Record<
		string,
		{
			version: string;
		}
	>;
}
