// Tests that the local network supports UDP communication in the way it's used for
// playback synchronization and other features.
const assert = require('assert');
const dgram = require('dgram');

const PORT = 60539;

/** @param {number} timeout */
function wait(timeout) {
	return new Promise((resolve) => setTimeout(resolve, timeout));
}

function createSocket() {
	const socket = dgram.createSocket({
		type: 'udp4',
		reuseAddr: true,
	});

	// socket.addListener('message', handleMessage);
	socket.bind(PORT);

	return socket;
}

/**
 * @param {import("dgram").Socket} socket
 */
async function open(socket) {
	await new Promise((resolve) => {
		socket.once('listening', () => {
			socket.setBroadcast(true);
			resolve(undefined);
		});
	});
}

/**
 * @param {import("dgram").Socket} socket
 * @param {string} message
 */
function sendMessage(socket, message) {
	const messageBuffer = Buffer.from(message);

	return new Promise((resolve, reject) => {
		socket.send(messageBuffer, 0, messageBuffer.length, PORT, '255.255.255.255', (/** @type {any} */ error) => {
			if (error) {
				reject(error);
			} else {
				resolve(undefined);
			}
		});
	});
}

/**
 * @param {import("dgram").Socket} socket
 */
function waitForMessage(socket) {
	return new Promise((resolve) => {
		socket.once('message', (messageBuffer) => {
			resolve(messageBuffer.toString());
		});
	});
}

(async () => {
	const socket = createSocket();
	await open(socket);

	console.log('Testing UDP communication');

	for (let i = 0; i < 10000; i++) {
		const message = 'message' + i;
		const responsePromise = waitForMessage(socket);
		await sendMessage(socket, message);
		const response = await responsePromise;
		assert.strictEqual(response, message);

		// it has to be artificially slowed down to avoid some weird race conditions
		await wait(5);

		if (i > 0 && i % 100 === 0) {
			console.log('Sent ' + i + ' messages');
		}
	}

	socket.close();
	console.log('OK');
})();
