import { getIndexedDBPath } from './FileSystem/IndexedDB/indexedDBHelper';
import { CacheName } from './ServiceWorker/Cache';

// This only happens once, when the browser sees this version of the ServiceWorker for the first time.
self.addEventListener('install', function (event: ExtendableEvent) {
	console.info('Installing service worker', event);
	event.waitUntil(
		caches.open(CacheName.FrontDisplay).then(function (cache) {
			return cache.addAll(['bundle.js', 'styles.css']);
		}),
	);
});

self.addEventListener('activate', function (event: ExtendableEvent) {
	console.info('Claiming control', event);
	event.waitUntil(self.clients.claim());
});

// The fetch event happens for the page request with the ServiceWorker's scope, and any request made within that page
self.addEventListener('fetch', async function (event: FetchEvent) {
	event.respondWith(handleFetchRequest(event));
});

export async function handleFetchRequest(event: FetchEvent) {
	const indexedDBCachedPath = getIndexedDBPath(event.request.url);
	if (!indexedDBCachedPath) {
		const fetchedResponse = await fetch(event.request);
		return fetchedResponse;
	}

	const cachedResponse = await caches.match(indexedDBCachedPath, { ignoreSearch: true });
	if (!cachedResponse) {
		return new Response('Not found', { status: 404, statusText: 'Not found' });
	}

	return cachedResponse;
}
