import front from './Front/front';
import management from './Management/management';
import Raven from 'raven-js';
import { useRavenLogging } from './Logging/logger';
import { createSocketSynchronizer } from './Synchronization/SocketSynchronizer/socketSynchronizerFactory';
import { createSameThreadWebWorkerFactory } from './WebWorker/masterWebWorkerFactory';
import { createDefaultDrivers } from './NativeDevice/Default/defaultDriverFactory';
import { parameters } from './options';
import { SubscriptionType } from './Display/IConfig';
import { registerServiceWorker } from './ServiceWorker/serviceWorkerLoader';

if (parameters.raven) {
	Raven.config(parameters.raven.dsn, parameters.raven.config).install();
}
useRavenLogging(window);

(async () => {
	const serviceWorkerRegistration = await registerServiceWorker(window);
	if (parameters.bundledApplet !== null && !parameters.autoVerification?.organizationUid) {
		throw new Error('sOS Open requires organization uid to be set during build.'); // TODO add more info, link, etc.
	}
	const autoVerification = parameters.autoVerification ? { organizationUid: parameters.autoVerification.organizationUid } : undefined;
	const subscriptionType = getSubscriptionType();

	const { frontDriver, managementDriver, cacheDriver } = createDefaultDrivers(window, serviceWorkerRegistration?.scope);
	await frontDriver.initialize(parameters.url.staticBaseUrl);
	await managementDriver.initialize(parameters.url.staticBaseUrl);
	const synchronizer = createSocketSynchronizer(parameters.url.synchronizerServerUrl, () => frontDriver);

	try {
		const deviceUid = await managementDriver.getDeviceUid();
		Raven.setUserContext({
			id: deviceUid,
		});
	} catch (error) {
		console.error(error);
	}

	const webWorkerFactory = createSameThreadWebWorkerFactory();

	const commonOptions = {
		global: window,
		baseUrl: parameters.url.baseUrl,
		platformUri: parameters.url.socketUri,
		staticBaseUrl: parameters.url.staticBaseUrl,
		uploadBaseUrl: parameters.url.uploadBaseUrl,
		checkInterval: parameters.app.checkIntervalMs,
		publicKey: parameters.httpAuthPublicKey,
		sessionIdKey: parameters.app.sessionIdKey,
		frontAppletPrefix: parameters.frontApplet.prefix,
		frontDisplayVersion: parameters.app.version,
		weinreServerUrl: parameters.url.weinreServerUrl,
		pingUrl: parameters.url.pingUrl,
		extendedManagementUrl: null,
		nativeDriver: { frontDriver, cacheDriver, managementDriver },
		managementDriver: managementDriver,
		synchronizer,
		webWorkerFactory,
		applicationVersion: getApplicationVersion(),
		bundledApplet: parameters.bundledApplet,
		autoVerification,
		excludedResponsibilities: [],
		subscriptionType,
		bundledServlet: null,
	};
	const frontOptions = { ...commonOptions, shortAppletFilesUrl: false };

	await front(frontOptions);
	await management(commonOptions);
})().catch((error: any) => console.error(error));

function getApplicationVersion() {
	const PATHNAME_REGEXP = /\/((\d+\.\d+\.\d+)(-[^\/]+)?)\/index\.html$/;
	const path = window.location.pathname;
	const matches = path.match(PATHNAME_REGEXP);
	if (!matches) {
		return parameters.app.version;
	}
	return matches[1];
}

function getSubscriptionType(): SubscriptionType {
	if (parameters.subscriptionType && parameters.subscriptionType in SubscriptionType) {
		return parameters.subscriptionType as SubscriptionType;
	}
	return SubscriptionType.open;
}
