import { delay, CallEffect, SagaReturnType } from 'redux-saga/effects';

type AsyncCallReturn<TFn extends (...params: any[]) => any> = SagaReturnType<TFn> | CallEffect<true>;

/**
 * Forwarding results of generator or async generator to parent Saga by iterating over it and
 * if it returns Promise instead of value returns Delay effect until the Promise will be resolved.
 * Supports Redux-Saga effects at all.
 */
export default function* asyncCall<Fn extends (...params: any[]) => any>(
	fn: Fn,
	...params: Parameters<Fn>
): Generator<Parameters<Fn> | AsyncCallReturn<Fn>, AsyncCallReturn<Fn>, AsyncCallReturn<Fn>> {
	const iterator = fn(...params);
	if (typeof iterator.next !== 'function') {
		return iterator;
	}
	while (true) {
		const resultOrPromise = iterator.next();
		if (typeof resultOrPromise.then === 'function') {
			let resolved = false,
				finalResult: IteratorResult<any> | undefined = undefined,
				finalException = undefined;
			resultOrPromise
				.then((result: any) => {
					finalResult = result;
				})
				.catch((e: Error) => {
					finalException = e;
				})
				.finally(() => {
					resolved = true;
				});
			while (!resolved) {
				yield delay(10);
			}
			if (finalException) {
				throw finalException;
			} else if (finalResult!.done!) {
				return finalResult!.value;
			}
			yield finalResult!.value;
			continue;
		} else if (resultOrPromise.done) {
			return resultOrPromise.value;
		}
		yield resultOrPromise.value;
	}
}
