import { ProprietaryTimerType } from '@signageos/common-types/dist/Device/Timer/ProprietaryTimerType';
import { NativeTimerType } from '@signageos/common-types/dist/Device/Timer/NativeTimerType';
import { ILegacyTimer } from '@signageos/lib/dist/Timer/Power/PowerTimerSettings';
import CommonTimerType from '@signageos/lib/dist/Timer/Power/PowerTimerType';
import { IPropertyStorage } from '../Property/propertyStorage';
import ITimerStorage, { TimerSettingsInternalStorage, TimersSettingsObject } from './ITimerStorage';
import Property from '../Property/Property';
import { IShortWeekdayTimer } from '@signageos/common-types/dist/Device/Settings/DeviceSettings';
import { convertWeekday } from '@signageos/lib/dist/Timer/convertWeekday';

export default class TimerPropertyStorage<T extends IShortWeekdayTimer, TKey extends ProprietaryTimerType | NativeTimerType>
	implements ITimerStorage<TKey, T>
{
	constructor(
		private propertyStorage: IPropertyStorage,
		private property: Property,
	) {}

	/** @deprecated use replaceTimerSettings instead */
	public async replaceTimers(newTimers: TimersSettingsObject): Promise<void> {
		await this.replaceShortTimers(this.convertTimers(newTimers));
	}

	public async replaceShortTimers(timers: Partial<Record<TKey, T>>): Promise<void> {
		await this.saveTimersToStorage({ ...timers });
	}

	/** @deprecated use setTimerSettings instead */
	public async setTimer(newTimer: ILegacyTimer): Promise<void> {
		await this.setShortTimer(this.convertTimer(newTimer));
	}

	public async setShortTimer(timer: T): Promise<void> {
		await this.saveTimersToStorage({
			...this.convertTimers(await this.retrieveTimersFromStorage()),
			[timer.type]: timer,
		});
	}

	/** @deprecated use getTimerSettings instead */
	public async getTimer(timerType: CommonTimerType): Promise<ILegacyTimer | undefined> {
		const timers = await this.getTimers();
		return timers[timerType];
	}

	public async getShortTimer(timerType: TKey): Promise<T | undefined> {
		let timers = await this.getShortTimers();
		return timers[timerType];
	}

	/** @deprecated use getTimerSettings instead */
	public async getTimers(): Promise<TimersSettingsObject> {
		return this.convertToDeprecatedTimers(await this.retrieveTimersFromStorage());
	}

	public async getShortTimers(): Promise<Record<TKey, T>> {
		return this.convertTimers(await this.retrieveTimersFromStorage());
	}

	/** @deprecated use deleteTimerSettings instead */
	public async deleteTimer(timerType: TKey): Promise<void> {
		return this.deleteShortTimer(timerType);
	}

	public async deleteShortTimer(timerType: TKey): Promise<void> {
		const timers = await this.getShortTimers();
		delete timers[timerType];
		this.saveTimersToStorage(timers);
	}

	private async saveTimersToStorage(timers: Partial<Record<TKey, T>>) {
		await this.propertyStorage.setValue(this.property, timers);
	}

	private async retrieveTimersFromStorage() {
		return this.propertyStorage.getValueOrDefault<TimerSettingsInternalStorage>(this.property, {});
	}

	private convertTimers(timers: TimerSettingsInternalStorage): Record<TKey, T> {
		return Object.fromEntries(
			Object.entries<T | ILegacyTimer>(timers as Record<TKey, T | ILegacyTimer>).map(([k, v]) => [
				k,
				'weekdays' in v ? v : this.convertTimer(v),
			]),
		) as Record<TKey, T>;
	}

	private convertTimer(t: ILegacyTimer): T {
		const { sun, mon, tue, wed, thu, fri, sat, ...rest } = t;
		return {
			...(rest as unknown as T),
			weekdays: convertWeekday.fromLegacy({ sun, mon, tue, wed, thu, fri, sat }).toShort(),
		};
	}

	/** @deprecated for BC only */
	private convertToDeprecatedTimers(timers: TimerSettingsInternalStorage): TimersSettingsObject {
		return Object.fromEntries(
			Object.entries<T | ILegacyTimer>(timers as Record<TKey, T | ILegacyTimer>).map(([k, v]) => [
				k,
				'weekdays' in v ? this.convertToDeprecatedTimer(v) : v,
			]),
		) as TimersSettingsObject;
	}

	/** @deprecated for BC only */
	private convertToDeprecatedTimer({ weekdays, ...rest }: T) {
		return {
			...rest,
			...convertWeekday.fromShort(weekdays).toLegacy(),
		};
	}
}
