import { ProprietaryTimer } from '@signageos/common-types/dist/Device/Settings/DeviceSettings';
import { ProprietaryTimerType } from '@signageos/common-types/dist/Device/Timer/ProprietaryTimerType';
import { now as getCurrentMoment } from '@signageos/lib/dist/DateTime/dateTimeFactory';
import { getLastTimerEvent } from '@signageos/lib/dist/Timer/Power/powerTimerComputer';
import wait from '@signageos/lib/dist/Timer/wait';
import Debug from 'debug';
import { EventEmitter } from 'events';
import IProprietaryResolver, { TimerEvent, TimerListener } from './IProprietaryTimerResolver';
import { IProprietaryTimerStorage } from './ITimerStorage';

const debug = Debug('@signageos/front-display:Timer:ProprietaryTimerManager');

const getCurrentDate = () => getCurrentMoment().toDate();

export default class ProprietaryTimerResolver implements IProprietaryResolver<TimerEvent> {
	private readonly eventEmitter: EventEmitter = new EventEmitter();

	private hasStartedListening: boolean = false;

	constructor(
		private timersStorage: IProprietaryTimerStorage,
		private readonly checkPeriod: number = 30 * 1e3,
		private now: () => Date = getCurrentDate,
		private appletLastEnabled: boolean = true,
	) {
		// We expect exactly a single TimerEvent.TIMER_ON and a single TimerEvent.TIMER_OFF listener.
		this.eventEmitter.setMaxListeners(2);
	}

	public on(event: TimerEvent, listener: TimerListener): this {
		this.eventEmitter.on(event, listener);

		if (!this.hasStartedListening) {
			this.hasStartedListening = true;
			this.timersChecking();
			// TODO Add cancelation mechanism and clean up after use.
		}

		return this;
	}

	private emit(event: TimerEvent) {
		this.eventEmitter.emit(event);
	}

	private async timersChecking() {
		while (true) {
			debug('check timers now');
			try {
				const now = this.now();
				await checkTimer(
					now,
					() => this.timersStorage.getShortTimers(),
					(event: TimerEvent) => this.emit(event),
					(state: boolean) => this.setLastAppletState(state),
					this.appletLastEnabled,
				);
			} catch (error) {
				console.error('timerChecking failed', error);
			}
			await wait(this.checkPeriod);
		}
	}
	private async setLastAppletState(lastState: boolean) {
		this.appletLastEnabled = lastState;
		return this.appletLastEnabled;
	}
}

export async function checkTimer(
	now: Date,
	getTimers: () => Promise<Record<ProprietaryTimerType, ProprietaryTimer>>,
	emit: (event: TimerEvent) => void,
	setLastAppletState: (state: boolean) => Promise<boolean>,
	appletLastEnabled: boolean,
) {
	const timers = await getTimers();
	const lastTimerEvent = getLastTimerEvent(timers, now);
	if (lastTimerEvent == null) {
		return;
	}

	const isTimerOn = lastTimerEvent.type === 'ON';
	const displayOn = appletLastEnabled;

	if (isTimerOn !== displayOn) {
		if (isTimerOn) {
			debug('emit timer on');
			await setLastAppletState(true);
			emit(TimerEvent.TIMER_ON);
		}

		if (!isTimerOn && !timers[lastTimerEvent.timerType].keepAppletRunning) {
			await setLastAppletState(false);
			debug('emit timer off');
			emit(TimerEvent.TIMER_OFF);
		}
	}
}
