import { EventEmitter } from 'events';
import Debug from 'debug';
import wait from '@signageos/lib/dist/Timer/wait';
import IProprietaryResolver, { TimerEvent, TimerListener, DisplayEvent } from './IProprietaryTimerResolver';
import { IProprietaryTimerStorage } from './ITimerStorage';
import { getLastTimerEvent } from '@signageos/lib/dist/Timer/Power/powerTimerComputer';
import { now as getCurrentMoment } from '@signageos/lib/dist/DateTime/dateTimeFactory';

const debug = Debug('@signageos/front-display:Timer:DisplayPowerTimerResolver');

const getCurrentDate = () => getCurrentMoment().toDate();

export default class DisplayPowerTimerResolver implements IProprietaryResolver<DisplayEvent> {
	private readonly eventEmitter: EventEmitter = new EventEmitter();

	private hasStartedListening: boolean = false;

	constructor(
		private getIsDisplayPowerOn: () => Promise<boolean>,
		private ProprietaryTimerResolver: IProprietaryResolver<TimerEvent>,
		private readonly checkPeriod: number = 30 * 1e3,
		private timersStorage: IProprietaryTimerStorage,
		private now: () => Date = getCurrentDate,
	) {
		// We expect exactly a single DisplayEvent.POWER_ON and a single DisplayEvent.POWER_OFF listener.
		this.eventEmitter.setMaxListeners(2);
	}
	public on(event: DisplayEvent, listener: TimerListener): this {
		this.eventEmitter.on(event, listener);

		if (!this.hasStartedListening) {
			this.hasStartedListening = true;
			this.displayEventEmitter();
			// TODO Add cancelation mechanism and clean up after use.
		}

		return this;
	}

	private emit(event: DisplayEvent) {
		this.eventEmitter.emit(event);
	}

	private async displayEventEmitter() {
		this.ProprietaryTimerResolver.on(TimerEvent.TIMER_ON, async () => {
			const displayOn = await this.getIsDisplayPowerOn();
			debug('emit display On');
			if (!displayOn) {
				this.emit(DisplayEvent.POWER_ON);
			}
		});

		this.ProprietaryTimerResolver.on(TimerEvent.TIMER_OFF, async () => {
			debug('emit display Off');
			const displayOn = await this.getIsDisplayPowerOn();
			if (displayOn) {
				this.emit(DisplayEvent.POWER_OFF);
			}
		});

		while (true) {
			let lastTimerEvent = getLastTimerEvent(await this.timersStorage.getShortTimers(), this.now());
			if (lastTimerEvent === null) {
				return;
			}

			let isTimerOn = lastTimerEvent.type === 'ON';
			const displayOn = await this.getIsDisplayPowerOn();

			if (isTimerOn !== displayOn) {
				if (isTimerOn) {
					this.emit(DisplayEvent.POWER_ON);
				} else {
					this.emit(DisplayEvent.POWER_OFF);
				}
			}

			await wait(this.checkPeriod);
		}
	}
}
