/* eslint-disable @typescript-eslint/no-shadow */
import { TestResult, ItemResult, TestCaseResult } from '@signageos/actions/dist/Device/Test/deviceTestActions';
import { withTimeout } from '@signageos/lib/dist/Timer/timeout';
import Debug from 'debug';

const debug = Debug('@signageos/front-display:Test:TestFramework');

const DEFAULT_TIMEOUT = 120e3;

enum TestItemType {
	TITLE = 'Title',
	DESCRIBE = 'Describe',
	TEST = 'Test',
	BEFORE = 'Before',
	BEFORE_EACH = 'BeforeEach',
	AFTER = 'After',
	AFTER_EACH = 'AfterEach',
}

export interface TestItem<TRun> {
	type: TestItemType;
	title: string;
	run: () => TRun;
	timeout?: number;
}

class SkipError extends Error {
	constructor(reason: string) {
		super(reason);
		// https://github.com/Microsoft/TypeScript/wiki/Breaking-Changes#extending-built-ins-like-error-array-and-map-may-no-longer-work
		Object.setPrototypeOf(this, SkipError.prototype);
	}
}

export class Describe implements TestItem<IterableIterator<TestItem<any>>> {
	public type: TestItemType = TestItemType.DESCRIBE;

	constructor(
		public title: string,
		public run: () => IterableIterator<TestItem<any>>,
		public timeout?: number,
	) {}
}

export function describe(title: string, describeCallback: () => IterableIterator<TestItem<any>>, timeout?: number) {
	return new Describe(title, describeCallback, timeout);
}

class It implements TestItem<Promise<void>> {
	public type: TestItemType = TestItemType.TEST;
	constructor(
		public title: string,
		public run: () => Promise<void>,
		public timeout?: number,
	) {}
}

export function it(title: string, describeCallback: () => Promise<void>, timeout?: number) {
	return new It(title, describeCallback, timeout);
}

export function skip(reason: string) {
	throw new SkipError(reason);
}

class Before implements TestItem<Promise<void>> {
	public type: TestItemType = TestItemType.BEFORE;
	constructor(
		public title: string,
		public run: () => Promise<void>,
		public timeout?: number,
	) {}
}

export function before(title: string, describeCallback: () => Promise<void>, timeout?: number) {
	return new Before(title, describeCallback, timeout);
}

class After implements TestItem<Promise<void>> {
	public type: TestItemType = TestItemType.AFTER;
	constructor(
		public title: string,
		public run: () => Promise<void>,
		public timeout?: number,
	) {}
}

export function after(title: string, describeCallback: () => Promise<void>, timeout?: number) {
	return new After(title, describeCallback, timeout);
}
class BeforeEach implements TestItem<Promise<void>> {
	public type: TestItemType = TestItemType.BEFORE_EACH;
	constructor(
		public title: string,
		public run: () => Promise<void>,
		public timeout?: number,
	) {}
}

export function beforeEach(title: string, describeCallback: () => Promise<void>, timeout?: number) {
	return new BeforeEach(title, describeCallback, timeout);
}

class AfterEach implements TestItem<Promise<void>> {
	public type: TestItemType = TestItemType.AFTER_EACH;
	constructor(
		public title: string,
		public run: () => Promise<void>,
		public timeout?: number,
	) {}
}

export function afterEach(title: string, describeCallback: () => Promise<void>, timeout?: number) {
	return new AfterEach(title, describeCallback, timeout);
}

type TestSuite = {
	describes: TestItem<IterableIterator<TestItem<any>>>[];
	tests: TestItem<Promise<void>>[];
	before: TestItem<Promise<void>>[];
	after: TestItem<Promise<void>>[];
	beforeEach: TestItem<Promise<void>>[];
	afterEach: TestItem<Promise<void>>[];
};

function loadTestSuite(testPayload: IterableIterator<TestItem<any>>): TestSuite {
	const creatingTestSuite: TestSuite = { tests: [], describes: [], before: [], after: [], beforeEach: [], afterEach: [] };
	for (let item of testPayload) {
		switch (item.type) {
			case TestItemType.TEST:
				creatingTestSuite.tests.push(item);
				break;
			case TestItemType.DESCRIBE:
				creatingTestSuite.describes.push(item);
				break;
			case TestItemType.BEFORE:
				creatingTestSuite.before.push(item);
				break;
			case TestItemType.AFTER:
				creatingTestSuite.after.push(item);
				break;
			case TestItemType.BEFORE_EACH:
				creatingTestSuite.beforeEach.push(item);
				break;
			case TestItemType.AFTER_EACH:
				creatingTestSuite.afterEach.push(item);
				break;
			default:
				throw new Error(`Unknown TestItemType: ${item.type}`);
		}
	}
	return creatingTestSuite;
}

export async function runTests(suite: Describe): Promise<TestResult> {
	const testSuite = loadTestSuite(suite.run());
	const result = createEmptyTestResult(suite.title);

	// describe - BEFORE
	// eslint-disable-next-line @typescript-eslint/no-shadow
	const { before, failed, skipped } = await runBefore(testSuite.before);
	result.before = before;

	if (failed || skipped) {
		result.total++;

		if (failed) {
			result.failed++;
		}
		if (skipped) {
			result.skipped++;
		}

		return result;
	}

	// describe - nested Describe
	for (let describe of testSuite.describes) {
		const describeResult = await runTests(describe);
		result.describe.push(describeResult);

		result.total += describeResult.total;
		result.successful += describeResult.successful;
		result.skipped += describeResult.skipped;
		result.failed += describeResult.failed;
		result.duration += describeResult.duration;
	}

	// describe - tests
	const tests = await runActualTests(testSuite);
	result.test = tests;

	for (let test of tests) {
		result.total++;
		result.duration += test.test.duration;

		let beforeEachFailed = false;
		let beforeEachSkipped = false;
		for (let beforeEach of test.beforeEach) {
			if (beforeEach.failed) {
				beforeEachFailed = true;
			}
			if (beforeEach.skipped) {
				beforeEachSkipped = true;
			}
		}

		if (beforeEachFailed) {
			result.failed++;
			continue;
		}
		if (beforeEachSkipped) {
			result.skipped++;
			continue;
		}

		if (test.test.failed) {
			result.failed++;
			continue;
		}
		if (test.test.skipped) {
			result.skipped++;
			continue;
		}

		let afterEachFailed = false;
		let afterEachSkipped = false;
		for (let afterEach of test.afterEach) {
			if (afterEach.failed) {
				afterEachFailed = true;
			}
			if (afterEach.skipped) {
				afterEachSkipped = true;
			}
		}

		if (afterEachFailed) {
			result.failed++;
			continue;
		}
		if (afterEachSkipped) {
			result.skipped++;
			continue;
		}

		// consider test being successful - neither skip nor failure
		result.successful++;
	}

	// describe - after - cleaning - failures/skips in here do not modify the overall test results
	result.after = await runItems(testSuite.after);

	return result;
}

export function createEmptyTestResult(title: string): TestResult {
	return {
		title,
		total: 0,
		successful: 0,
		failed: 0,
		skipped: 0,
		test: [],
		describe: [],
		before: [],
		after: [],
		duration: 0,
	};
}

async function runActualTests(testSuite: TestSuite): Promise<
	{
		beforeEach: ItemResult[];
		test: ItemResult;
		afterEach: ItemResult[];
	}[]
> {
	const testResults: TestCaseResult[] = [];

	for (let test of testSuite.tests) {
		const testResult: ItemResult = { title: test.title, skipped: false, failed: false, duration: 0 };

		let shouldPreventRun = false;
		const beforeEachResults = await runItems(testSuite.beforeEach);
		for (const beforeEach of beforeEachResults) {
			if (beforeEach.failed) {
				testResult.failed = true;
				testResult.reason = `Before each error. Reason: ${beforeEach.reason}`;
				shouldPreventRun = true;
			}
			if (beforeEach.skipped) {
				testResult.skipped = true;
				testResult.reason = `Before each skipped. Reason: ${beforeEach.reason}`;
				shouldPreventRun = true;
			}
		}

		if (shouldPreventRun) {
			testResults.push({ beforeEach: beforeEachResults, test: testResult, afterEach: [] });
			continue;
		}

		const timeStart = new Date().getTime();
		try {
			await withTimeout(test.run(), test.timeout || DEFAULT_TIMEOUT);
		} catch (error) {
			testResult.reason = error.message;
			if (error instanceof SkipError) {
				testResult.skipped = true;
			} else {
				testResult.failed = true;
			}
		}

		testResult.duration = new Date().getTime() - timeStart;
		const afterEachResult = await runItems(testSuite.afterEach);
		testResults.push({ beforeEach: beforeEachResults, test: testResult, afterEach: afterEachResult });
	}

	return testResults;
}

async function runBefore(items: TestItem<Promise<void>>[]): Promise<{ before: ItemResult[]; failed: boolean; skipped: boolean }> {
	const itemResults: ItemResult[] = [];
	for (let item of items) {
		const result = await executeItem(item);
		itemResults.push(result);

		if (result.skipped) {
			// skip all following on skip
			return { before: itemResults, failed: false, skipped: true };
		}

		if (result.failed) {
			// skip all following on failure
			return { before: itemResults, failed: true, skipped: true };
		}
	}

	return { before: itemResults, failed: false, skipped: false };
}

async function runItems(items: TestItem<Promise<void>>[]): Promise<ItemResult[]> {
	const itemResults: ItemResult[] = [];
	for (let item of items) {
		itemResults.push(await executeItem(item));
	}
	return itemResults;
}

async function executeItem(instance: TestItem<Promise<void>>): Promise<ItemResult> {
	const result: ItemResult = { title: instance.title, skipped: false, failed: false, duration: 0 };
	try {
		debug('run test: ' + instance.title);
		await withTimeout(instance.run(), instance.timeout || DEFAULT_TIMEOUT);
		debug('test successful: ' + instance.title);

		return result;
	} catch (error) {
		result.reason = error.message;

		if (error instanceof SkipError) {
			debug('test skipped: ' + instance.title);
			result.skipped = true;
			return result;
		}

		debug(`test failed: ${instance.title}, reason: ${error.message}`);
		result.failed = true;
		return result;
	}
}
