import { P2PSynchronizer } from './P2PSynchronizer';
import { PeerDiscoveryService } from '../../PeerNetwork/PeerDiscoveryService/PeerDiscoveryService';
import { ITcpSocket } from '../../PeerNetwork/Socket/Tcp/ITcpSocket';
import { ISerializer } from '../../PeerNetwork/Serializer/serialization';
import { IUdpSocket } from '../../PeerNetwork/Socket/Udp/IUdpSocket';
import { MessageTcpSocket } from '../../PeerNetwork/Socket/Tcp/MessageTcpSocket';
import { MessageUdpSocket } from '../../PeerNetwork/Socket/Udp/MessageUdpSocket';
import { NetworkInfo } from '../../PeerNetwork/PeerDiscoveryService/INetworkInfo';
import { PeerNetwork } from '../../PeerNetwork/PeerNetwork';
import { OrderedTcpSocket } from '../../PeerNetwork/Socket/Tcp/OrderedTcpSocket';

export const UDP_SYNC_PORT = 60537;
export const TCP_SYNC_PORT = 60538;

export type UdpSocketFactory<TSerialized> = (port: number) => Promise<IUdpSocket<TSerialized>>;
export type TcpSocketFactory<TSerialized> = (port: number) => Promise<ITcpSocket<TSerialized>>;

/**
 * Convenience factory to create and instance of P2PSynchronizer
 *
 * It uses PeerDiscoveryService and a default or custom socket factory.
 * By default it uses UdpSocket but a custom socket factory can be provided.
 */
export async function createP2PSynchronizer<TSerialized>({
	udpPort = UDP_SYNC_PORT,
	tcpPort = TCP_SYNC_PORT,
	getDeviceUid,
	getLocalAddress,
	serializer,
	udpSocketFactory,
	tcpSocketFactory,
}: {
	/**
	 * Port for UDP communication
	 * UDP communication is used for network discovery, i.e. to find other devices on the network
	 */
	udpPort?: number;
	/**
	 * Port for TCP communication
	 * TCP communication is used for sending messages between devices
	 */
	tcpPort?: number;
	getDeviceUid: () => Promise<string>;
	getLocalAddress: () => Promise<string | null>;
	serializer: ISerializer<TSerialized>;
	udpSocketFactory: UdpSocketFactory<TSerialized>;
	tcpSocketFactory: TcpSocketFactory<TSerialized>;
}) {
	const id = await getDeviceUid();
	const udpSocketFactoryWithoutPort = getUdpSocketFactoryWithoutPort(udpSocketFactory, serializer, udpPort);
	const tcpSocketFactoryWithoutPort = getTcpSocketFactoryWithoutPort(tcpSocketFactory, serializer, tcpPort);
	const getNetworkInfo = getNetworkInfoProvider(tcpPort, getLocalAddress);
	const peerDiscoveryService = new PeerDiscoveryService(id, udpSocketFactoryWithoutPort, getNetworkInfo);
	const peerNetwork = new PeerNetwork(peerDiscoveryService, tcpSocketFactoryWithoutPort);
	return new P2PSynchronizer(peerNetwork);
}

function getNetworkInfoProvider(port: number, getLocalAddress: () => Promise<string | null>): () => Promise<NetworkInfo | null> {
	return async () => {
		const address = (await getLocalAddress()) ?? '0.0.0.0';
		return {
			address,
			port,
		};
	};
}

function getTcpSocketFactoryWithoutPort<TSerialized>(
	socketFactory: TcpSocketFactory<TSerialized>,
	serializer: ISerializer<TSerialized>,
	port: number,
) {
	return async () => {
		const socket = await socketFactory(port);
		const orderedSocket = new OrderedTcpSocket(socket);
		return new MessageTcpSocket(orderedSocket, serializer);
	};
}

function getUdpSocketFactoryWithoutPort<TSerialized>(
	socketFactory: UdpSocketFactory<TSerialized>,
	serializer: ISerializer<TSerialized>,
	port: number,
) {
	return async () => {
		const socket = await socketFactory(port);
		return new MessageUdpSocket(socket, serializer);
	};
}
