export enum SynchronizerEvent {
	GroupStatus = 'group_status',
	GroupLeft = 'group_left',
	BroadcastedValue = 'broadcasted_value',
	Closed = 'closed',
}

export interface GroupStatus {
	groupName: string;
	connectedPeers: string[];
	/**
	 * True if this client is currently master of the group
	 *
	 * Master is selected automatically by the synchronizer and can change at any moment.
	 *
	 * This is useful in situations where one device in the group should control the behavior of the entire group,
	 * i.e. triggering some action at a certain time.
	 *
	 * It's also possible to check if the device is master by calling isMaster method.
	 */
	isMaster: boolean;
}

export interface BroadcastedValue {
	groupName: string;
	key: string;
	value: unknown;
}

export type GroupStatusCallback = (deviceStatus: GroupStatus) => void;
export type GroupLeftCallback = (groupName: string) => void;
export type BroadcastedValueCallback = (broadcastedValue: BroadcastedValue) => void;
export type ClosedCallback = (error?: Error) => void;
export type SynchronizerEventCallback = GroupStatusCallback | GroupLeftCallback | BroadcastedValueCallback | ClosedCallback;

export interface JoinGroupArgs {
	groupName: string;
	deviceIdentification?: string;
}

export interface WaitArgs {
	groupName: string;
	data?: unknown;
	timeoutMs?: number;
}

export interface BroadcastedValueArgs {
	groupName: string;
	key: string;
	value: unknown;
}

/**
 * Connects with other devices and synchronizes with their runtime
 */
interface ISynchronizer {
	connect(serverUri?: string): Promise<void>;
	close(): Promise<void>;
	isConnected(): Promise<boolean>;
	/** Join a group and start communication with other peers in the same group */
	joinGroup(args: JoinGroupArgs): Promise<void>;
	/** Leave a previously joined group and stop all communication with other peers in that group */
	leaveGroup(groupName: string): Promise<void>;
	/**
	 * Get device identification that this device uses within a group
	 *
	 * If joinGroup was called with defined deviceIdentification, it will return that value.
	 * Otherwise, it will return a randomly generated value that was generated when called joinGroup.
	 *
	 * @returns device identification if joined the group, undefined otherwise
	 */
	getDeviceIdentification(groupName: string): Promise<string | undefined>;
	/**
	 * Wait for everybody else in the group to also call wait and resolve at the same time.
	 *
	 * The method accepts data argument. If used, when it resolves, one of the clients in the group
	 * will be selected as master and everybody will resolve with the master's data.
	 *
	 * When multiple clients want to get synchronized, they may be at different stages of the runtime.
	 * For example, if they're playing some playlist in sync, they might join at different times,
	 * so they'd want to play different items from the playlist.
	 * That's why each one can send its data that represents, what he's planning to do (like playlist item ID).
	 * Once everybody is waiting, only one of them will be selected as master and everybody will
	 * resolve with the master's data. Then, everybody has to respect it.
	 *
	 * For example, if I sent item-3 but got back item-5, it's my responsibility to fast forward to item-5 and resume there
	 * in order to synchronize with the master.
	 *
	 * There's also an option to set a timeout. In that case, it will resolve at most after the timeout
	 * and won't keep waiting for the others. That should be used as last resort since it will result
	 * in behavior that's not synchronized with the rest of the clients.
	 */
	wait(args: WaitArgs): Promise<unknown>;
	/**
	 * Cancels pending wait for a group
	 *
	 * If there is a pending wait, it will reject.
	 * All peers will be notified that this peer canceled the wait.
	 */
	cancelWait(groupName: string): Promise<void>;
	/**
	 * Broadcast a value to all peers within the group
	 */
	broadcastValue(args: BroadcastedValueArgs): Promise<void>;
	/**
	 * Check if this device is the master in the group
	 *
	 * This is useful in situations where one device in the group should control the behavior of the entire group,
	 * i.e. triggering some action at a certain time.
	 *
	 * Whenever the master changes, GroupStatus event is emitted with the information about the new master.
	 */
	isMaster(groupName: string): Promise<boolean>;
	addListener(event: SynchronizerEvent.GroupStatus, listener: GroupStatusCallback): void;
	addListener(event: SynchronizerEvent.GroupLeft, listener: GroupLeftCallback): void;
	addListener(event: SynchronizerEvent.BroadcastedValue, listener: BroadcastedValueCallback): void;
	/**
	 * Emits Closed event when the connection is closed for whatever reason
	 *
	 * When close() is called, it will emit Closed event without an error. That means it was a clean close.
	 * If the synchronizer closes unexpectedly, for example due to some network error, it will emit Closed with an appropriate error.
	 */
	addListener(event: SynchronizerEvent.Closed, listener: ClosedCallback): void;
	removeListener(event: SynchronizerEvent.GroupStatus, listener: GroupStatusCallback): void;
	removeListener(event: SynchronizerEvent.GroupLeft, listener: GroupLeftCallback): void;
	removeListener(event: SynchronizerEvent.BroadcastedValue, listener: BroadcastedValueCallback): void;
	removeListener(event: SynchronizerEvent.Closed, listener: ClosedCallback): void;
}

export default ISynchronizer;
