export interface GroupMember {
	id: string;
	name: string;
	joinedAt: Date;
}

export enum GroupEvent {
	MemberJoined = 'member_joined',
	MemberLeft = 'member_left',
	Data = 'data',
}

export interface Data<TData = unknown> {
	from: GroupMember;
	data: TData;
}

export type MemberCallback = (peer: GroupMember) => void;
export type DataCallback = (data: Data) => void;

/**
 * Group automatically handles membership in a group for synchronization purposes.
 */
export interface IGroup {
	getGroupName(): string;
	/**
	 * Get me as a member of the group
	 */
	getMe(): GroupMember;
	/**
	 * Get other members of the group (not me)
	 */
	getPeers(): GroupMember[];
	/**
	 * Get all members of the group, me and peers
	 */
	getAllMembers(): GroupMember[];
	/**
	 * Get master of the group
	 *
	 * Master is me or one of the peers who should decide for the group whenever needed
	 */
	getMaster(): GroupMember;
	/**
	 * Establish connection with the other peers in the group in the network
	 */
	join(): void;
	/**
	 * Leave the group and break communication with the peers in the network
	 */
	leave(): void;
	sendGroupDataMessage(data: unknown): Promise<void>;
	addListener(event: GroupEvent.MemberJoined, callback: MemberCallback): void;
	addListener(event: GroupEvent.MemberLeft, callback: MemberCallback): void;
	addListener(event: GroupEvent.Data, callback: DataCallback): void;
	removeListener(event: GroupEvent.MemberJoined, callback: MemberCallback): void;
	removeListener(event: GroupEvent.MemberLeft, callback: MemberCallback): void;
	removeListener(event: GroupEvent.Data, callback: DataCallback): void;
	removeAllListeners(event?: GroupEvent): void;
}
