import { EventEmitter } from 'events';
import Debug from 'debug';
import { Data, GroupEvent, GroupMember, IGroup } from './IGroup';
import { BroadcastEvent, IBroadcastService, ValueCallback } from './IBroadcastService';

const debug = Debug('@signageos/front-display:Synchronization:BroadcastService');

enum MessageType {
	BroadcastValue = 'broadcast_value',
}

interface BroadcastValueMessage {
	type: MessageType.BroadcastValue;
	key: string;
	value: unknown;
}

function isMessage(message: unknown): message is { type: string } {
	return typeof message === 'object' && message !== null && 'type' in message;
}

function isBroadcastValueMessage(message: unknown): message is BroadcastValueMessage {
	return isMessage(message) && message.type === MessageType.BroadcastValue;
}

/**
 * Used to broadcast values to all devices in the same network.
 *
 * It uses IGroup under the hood which provides functionality to communicate with other devices in the same group.
 */
export class BroadcastService implements IBroadcastService {
	private emitter: EventEmitter = new EventEmitter();

	constructor(private group: IGroup) {
		this.handleData = this.handleData.bind(this);
	}

	public start() {
		this.group.addListener(GroupEvent.Data, this.handleData);
	}

	public stop() {
		this.group.removeListener(GroupEvent.Data, this.handleData);
	}

	public async broadcastValue(key: string, value: unknown) {
		const message: BroadcastValueMessage = {
			type: MessageType.BroadcastValue,
			key,
			value,
		};

		this.logDebug('broadcast value', JSON.stringify(message));
		this.group.sendGroupDataMessage(message);
	}

	public addListener(event: BroadcastEvent.Value, listener: ValueCallback): void {
		this.emitter.addListener(event, listener);
	}

	public removeListener(event: BroadcastEvent.Value, listener: ValueCallback): void {
		this.emitter.removeListener(event, listener);
	}

	private handleData({ from, data }: Data) {
		this.logDebug('got data message from ' + from.id, JSON.stringify(data));

		if (isBroadcastValueMessage(data)) {
			this.handleBroadcastValueMessage(from, data);
		}
	}

	private handleBroadcastValueMessage(from: GroupMember, message: BroadcastValueMessage) {
		this.logDebug('got broadcast value message from ' + from.id, JSON.stringify(message));
		this.emitter.emit(BroadcastEvent.Value, message.key, message.value);
	}

	private logDebug(...args: any[]) {
		debug(this.group.getGroupName(), ...args);
	}
}
