import { EventEmitter } from 'events';
import IStreamPlayer, { IStreamOptions, IStreamPrepareOptions, ITrackInfo, TrackType } from './IStreamPlayer';
import {
	IStreamErrorEvent,
	IStreamClosedEvent,
	IStreamDisconnectedEvent,
	IStreamConnectedEvent,
	IStreamReconnectEvent,
	IStreamTracksChangedEvent,
} from './streamEvents';
import IStream from './IStream';
import { IVideoSrcArguments } from '../Video/IVideoSrcArguments';
import { StreamEventType } from './IStreamEvent';
import Debug from 'debug';

const DEFAULT_RECONNECT_INTERVAL_MS = 30e3;

const debug = Debug('@signageos/front-display:Stream:ReconnectStreamPlayer');

interface IStreamProperties {
	props: {
		uri: string;
		x: number;
		y: number;
		width: number;
		height: number;
		options?: IStreamOptions;
		tracks?: ITrackInfo[];
	};
	emitter: EventEmitter;
	stream?: IStream;
	reconnectTimeoutHandler?: NodeJS.Timer;
}

class ReconnectStreamPlayer implements IStreamPlayer {
	private streamsProperties: Record<string, IStreamProperties> = {};

	constructor(private streamPlayer: IStreamPlayer) {}

	public prepare(uri: string, x: number, y: number, width: number, height: number, options?: IStreamPrepareOptions): Promise<void> {
		debug('prepare', uri, x, y, width, height, options);
		return this.streamPlayer.prepare(uri, x, y, width, height, options);
	}

	public async play(uri: string, x: number, y: number, width: number, height: number, options?: IStreamOptions): Promise<IStream> {
		const streamProperties = this.getStreamProperties(uri, x, y, width, height, options);
		debug('play', streamProperties);

		if (options?.autoReconnectInterval && options.autoReconnectInterval < 10000) {
			throw new Error('Auto reconnect interval must be at least 10000 ms');
		}

		this.streamConnect(streamProperties);

		return streamProperties.emitter;
	}

	public async stop(uri: string, x: number, y: number, width: number, height: number): Promise<void> {
		const streamProperties = this.getStreamProperties(uri, x, y, width, height);
		debug('stop', streamProperties);
		streamProperties.stream?.removeAllListeners();
		this.cancelStreamReconnectSchedule(streamProperties);
		await this.streamPlayer.stop(uri, x, y, width, height);
		this.emitEvent('closed', streamProperties);
		this.clearStreamProperties(streamProperties);
	}

	public async pause(uri: string, x: number, y: number, width: number, height: number): Promise<void> {
		debug('pause', uri, x, y, width, height);
		await this.streamPlayer.pause(uri, x, y, width, height);
	}

	public async resume(uri: string, x: number, y: number, width: number, height: number): Promise<void> {
		debug('resume', uri, x, y, width, height);
		await this.streamPlayer.resume(uri, x, y, width, height);
	}

	public async clearAll(): Promise<void> {
		Object.keys(this.streamsProperties).forEach((key: string) => this.cancelStreamReconnectSchedule(this.streamsProperties[key]));
		await this.streamPlayer.clearAll();
		Object.keys(this.streamsProperties).forEach((key: string) => this.clearStreamProperties(this.streamsProperties[key]));
	}

	public async getTracks(videoId: IVideoSrcArguments): Promise<ITrackInfo[]> {
		return this.streamPlayer.getTracks(videoId);
	}

	public async selectTrack(videoId: IVideoSrcArguments, trackType: TrackType, groupId: string, trackIndex: number): Promise<void> {
		await this.streamPlayer.selectTrack(videoId, trackType, groupId, trackIndex);
	}

	public async resetTrack(videoId: IVideoSrcArguments, trackType: TrackType, groupId?: string): Promise<void> {
		await this.streamPlayer.resetTrack(videoId, trackType, groupId);
	}

	private async streamConnect(streamProperties: IStreamProperties) {
		debug('stream connect', streamProperties);
		const { uri, x, y, width, height, options } = streamProperties.props;

		try {
			streamProperties.stream = await this.streamPlayer.play(uri, x, y, width, height, options);

			streamProperties.stream.on('connected', () => {
				debug('connected', streamProperties);
				this.emitEvent('connected', streamProperties);
			});

			streamProperties.stream.once('error', () => {
				debug('error', streamProperties);
				this.emitEvent('error', streamProperties);
				this.emitEvent('disconnected', streamProperties);
				this.scheduleStreamReconnect(streamProperties);
			});

			streamProperties.stream.once('disconnected', async () => {
				debug('disconnected', streamProperties);
				this.emitEvent('disconnected', streamProperties);
				this.scheduleStreamReconnect(streamProperties);
			});

			streamProperties.stream.on('tracks_changed', (event: IStreamTracksChangedEvent) => {
				debug('tracks_changed', streamProperties, event);
				streamProperties.props.tracks = event.tracks;
				this.emitEvent('tracks_changed', streamProperties);
			});
		} catch {
			debug('error', streamProperties);
			this.scheduleStreamReconnect(streamProperties);
		}
	}

	private scheduleStreamReconnect(streamProperties: IStreamProperties) {
		const autoReconnectEnabled = streamProperties.props.options?.autoReconnect ?? false;
		if (autoReconnectEnabled) {
			const reconnectIntervalMs = streamProperties.props.options?.autoReconnectInterval ?? DEFAULT_RECONNECT_INTERVAL_MS;
			if (!streamProperties.reconnectTimeoutHandler) {
				streamProperties.reconnectTimeoutHandler = setTimeout(() => {
					delete streamProperties.reconnectTimeoutHandler;
					this.streamReconnect(streamProperties);
				}, reconnectIntervalMs);
			} else {
				console.warn('Reconnect is already scheduled', streamProperties.props);
			}
		}
	}

	private async streamReconnect(streamProperties: IStreamProperties): Promise<void> {
		const { uri, x, y, width, height } = streamProperties.props;
		this.emitEvent('reconnect', streamProperties);
		await this.streamPlayer.stop(uri, x, y, width, height);
		await this.streamConnect(streamProperties);
	}

	private getKeyForProperties(uri: string, x: number, y: number, width: number, height: number): string {
		return `${uri}_${x}_${y}_${width}_${height}`;
	}

	private getStreamProperties(
		uri: string,
		x: number,
		y: number,
		width: number,
		height: number,
		options?: IStreamOptions,
	): IStreamProperties {
		const propertiesKey = this.getKeyForProperties(uri, x, y, width, height);

		if (!this.streamsProperties.hasOwnProperty(propertiesKey)) {
			this.streamsProperties[propertiesKey] = {
				props: { uri, x, y, width, height, options },
				emitter: new EventEmitter(),
			};
			this.streamsProperties[propertiesKey].emitter.on('error', () => {
				/* making it safe */
			});
		}

		return this.streamsProperties[propertiesKey];
	}

	private cancelStreamReconnectSchedule(streamProperties: IStreamProperties): void {
		const { uri, x, y, width, height } = streamProperties.props;
		const propertiesKey = this.getKeyForProperties(uri, x, y, width, height);

		if (this.streamsProperties[propertiesKey]?.reconnectTimeoutHandler) {
			clearTimeout(this.streamsProperties[propertiesKey].reconnectTimeoutHandler);
			delete this.streamsProperties[propertiesKey].reconnectTimeoutHandler;
		}
	}

	private clearStreamProperties(streamProperties: IStreamProperties): void {
		const { uri, x, y, width, height } = streamProperties.props;
		const propertiesKey = this.getKeyForProperties(uri, x, y, width, height);

		if (this.streamsProperties.hasOwnProperty(propertiesKey)) {
			this.cancelStreamReconnectSchedule(streamProperties);
			this.streamsProperties[propertiesKey].emitter.removeAllListeners();
			delete this.streamsProperties[propertiesKey];
		}
	}

	private emitEvent(type: StreamEventType, streamProperties: IStreamProperties) {
		const { uri, x, y, width, height, options, tracks } = streamProperties.props;
		const { emitter } = streamProperties;
		switch (type) {
			case 'error':
				emitter.emit('error', {
					type: 'error',
					uri,
					x,
					y,
					width,
					height,
					protocol: options?.protocol,
					errorMessage: 'Stream playback failed',
				} as IStreamErrorEvent);
				break;
			case 'connected':
				emitter.emit('connected', {
					type: 'connected',
					uri,
					x,
					y,
					width,
					height,
				} as IStreamConnectedEvent);
				break;
			case 'disconnected':
				emitter.emit('disconnected', {
					type: 'disconnected',
					uri,
					x,
					y,
					width,
					height,
				} as IStreamDisconnectedEvent);
				break;
			case 'closed':
				emitter.emit('closed', {
					type: 'closed',
					uri,
					x,
					y,
					width,
					height,
				} as IStreamClosedEvent);
				break;
			case 'reconnect':
				emitter.emit('reconnect', {
					type: 'reconnect',
					uri,
					x,
					y,
					width,
					height,
				} as IStreamReconnectEvent);
				break;
			case 'tracks_changed':
				emitter.emit('tracks_changed', {
					type: 'tracks_changed',
					uri,
					x,
					y,
					width,
					height,
					tracks,
				} as IStreamTracksChangedEvent);
				break;
			default:
				break;
		}
	}
}

export default ReconnectStreamPlayer;
