import { EventEmitter } from 'events';
import IStreamPlayer, { IStreamOptions, IStreamPrepareOptions, ITrackInfo, TrackType } from './IStreamPlayer';
import { IStreamErrorEvent, IStreamClosedEvent, IStreamDisconnectedEvent, IStreamConnectedEvent } from './streamEvents';
import IStream from './IStream';
import IVideoPlayer from '../Video/IVideoPlayer';
import Debug from 'debug';
import { IVideoSrcArguments } from '../Video/IVideoSrcArguments';

const debug = Debug('@signageos/front-display:Stream:ProprietaryStreamPlayer');

class ProprietaryStreamPlayer implements IStreamPlayer {
	constructor(private video: IVideoPlayer) {}

	public prepare(uri: string, x: number, y: number, width: number, height: number, options?: IStreamPrepareOptions): Promise<void> {
		return this.video.prepare(uri, x, y, width, height, options);
	}

	public async play(uri: string, x: number, y: number, width: number, height: number, options: IStreamOptions): Promise<IStream> {
		const video = await this.video.play(uri, x, y, width, height);
		const stream = new EventEmitter();

		setTimeout(
			() =>
				stream.emit('connected', {
					type: 'connected',
					uri,
					x,
					y,
					width,
					height,
					protocol: options.protocol,
				} as IStreamConnectedEvent),
			200,
		);

		video.once('error', () => {
			stream.emit('error', {
				type: 'error',
				uri,
				x,
				y,
				width,
				height,
				protocol: options.protocol,
				errorMessage: 'Stream playback failed',
			} as IStreamErrorEvent);
		});
		video.once('stopped', () => {
			stream.emit('closed', {
				type: 'closed',
				uri,
				x,
				y,
				width,
				height,
				protocol: options.protocol,
			} as IStreamClosedEvent);
		});
		video.once('ended', () => {
			stream.emit('disconnected', {
				type: 'disconnected',
				uri,
				x,
				y,
				width,
				height,
				protocol: options.protocol,
			} as IStreamDisconnectedEvent);
		});

		return stream;
	}

	public async stop(uri: string, x: number, y: number, width: number, height: number): Promise<void> {
		await this.video.stop(uri, x, y, width, height);
	}

	public async pause(uri: string, x: number, y: number, width: number, height: number): Promise<void> {
		await this.video.pause(uri, x, y, width, height);
	}

	public async resume(uri: string, x: number, y: number, width: number, height: number): Promise<void> {
		await this.video.resume(uri, x, y, width, height);
	}

	public async clearAll(): Promise<void> {
		debug('proprietary stream has nothing to clear');
	}

	public async getTracks(_videoId: IVideoSrcArguments): Promise<ITrackInfo[]> {
		throw new Error('Method getTracks not implemented.');
	}

	public async selectTrack(_videoId: IVideoSrcArguments, _trackType: TrackType, _groupId: string, _trackIndex: number): Promise<void> {
		throw new Error('Method selectTrack not implemented.');
	}

	public async resetTrack(_videoId: IVideoSrcArguments, _trackType: TrackType, _groupId?: string): Promise<void> {
		throw new Error('Method resetTrack not implemented.');
	}
}

export default ProprietaryStreamPlayer;
