import { posix as path } from 'path';
import AsyncStorage from './Async/AsyncStorage';
import IFileSystem from '../NativeDevice/IFileSystem';
import { IStorageUnit, IFilePath } from '../NativeDevice/fileSystem';

export default class FileSystemStorage implements AsyncStorage {
	private static DIRNAME: string = '.localStorage';

	constructor(private fileSystem: IFileSystem) {}

	public async getItem(key: string): Promise<string | null> {
		const filePath = await this.getFilePath(key);
		if (await this.fileSystem.exists(filePath)) {
			return await this.fileSystem.readFile(filePath);
		} else {
			return null;
		}
	}

	public async setItem(key: string, data: string): Promise<void> {
		await this.ensureParentDir();
		const filePath = await this.getFilePath(key);
		await this.fileSystem.writeFile(filePath, data);
	}

	public async removeItem(key: string): Promise<void> {
		const filePath = await this.getFilePath(key);
		if (await this.fileSystem.exists(filePath)) {
			await this.fileSystem.deleteFile(filePath, false);
		}
	}

	public async clear(): Promise<void> {
		const parentDirPath = await this.getParentDirPath();
		await this.fileSystem.deleteFile(parentDirPath, true);
	}

	private async ensureParentDir() {
		const parentDirPath = await this.getParentDirPath();
		if (!(await this.fileSystem.exists(parentDirPath))) {
			await this.fileSystem.createDirectory(parentDirPath);
		}
	}

	private async getFilePath(key: string): Promise<IFilePath> {
		const internalStorageUnit = await this.getInternalStorageUnit();
		return {
			filePath: path.join(FileSystemStorage.DIRNAME, key),
			storageUnit: internalStorageUnit,
		};
	}

	private async getParentDirPath() {
		const internalStorageUnit = await this.getInternalStorageUnit();
		return {
			filePath: FileSystemStorage.DIRNAME,
			storageUnit: internalStorageUnit,
		};
	}

	private async getInternalStorageUnit() {
		const storageUnits = await this.fileSystem.listStorageUnits();
		const internalStorageUnit = storageUnits.find((storageUnit: IStorageUnit) => !storageUnit.removable);
		if (!internalStorageUnit) {
			throw new Error("Internal storage isn't available");
		}
		return internalStorageUnit;
	}
}
