const PREFIX = 'CategorizedLocalStorage';

export default class CategorizedLocalStorage implements Storage {
	[itemKey: string]: any;
	[itemIndex: number]: string;

	constructor(
		private storage: Storage,
		private category: string,
	) {
		Object.defineProperty(this, 'storage', { enumerable: false });
		Object.defineProperty(this, 'category', { enumerable: false });
		this.getKeys().forEach((key: string) => this.defineItem(key));
	}

	public get length() {
		return this.getKeys().length;
	}

	public clear() {
		for (let key of this.getKeys()) {
			this.removeItem(key);
		}
	}

	public getItem(key: string) {
		const value = this.storage.getItem(this.getKey(key));
		return typeof value !== 'undefined' ? value : null;
	}

	public removeItem(key: string) {
		this.storage.removeItem(this.getKey(key));
		this.undefineItem(key);
	}

	public setItem(key: string, data: string) {
		this.storage.setItem(this.getKey(key), '' + data);
		this.defineItem(key);
	}

	public key(index: number) {
		return this.getKeys()[index];
	}

	private getKeys() {
		return Object.keys(this.storage)
			.filter((key: string) => key.indexOf(this.getPrefix()) === 0)
			.map((key: string) => key.substring(this.getPrefix().length));
	}

	private getKey(key: string) {
		return this.getPrefix() + key;
	}

	private getPrefix() {
		return PREFIX + '$' + this.category + '$';
	}

	private defineItem(key: string) {
		Object.defineProperty(this, key, {
			configurable: true,
			enumerable: true,
			get: () => this.getItem(key),
		});
	}

	private undefineItem(key: string) {
		Object.defineProperty(this, key, {
			configurable: true,
			enumerable: false,
			writable: false,
			value: undefined,
		});
	}
}
