import IServletRunner, { IServletOptions, IServletProcess, IServletProcessMessage, IHealthcheckOptions } from './IServletRunner';
import { IFilePath } from '../NativeDevice/fileSystem';
import SupervisedServletProcess from './SupervisedServletProcess';
import Debug from 'debug';

const debug = Debug('@signageos/front-display:Servlet:SupervisedServletRunner');

const DEFAULT_HEALTHCHECK_TIMEOUT_MS = 30e3;

export default class SupervisedServletRunner implements IServletRunner {
	private servletProcesses: SupervisedServletProcess[] = [];

	constructor(
		private messageTypePrefix: string,
		private baseServletRunner: IServletRunner,
		private defaultHealthcheckTimeoutMs: number = DEFAULT_HEALTHCHECK_TIMEOUT_MS,
	) {}

	public run(entryPoint: IFilePath, options?: IServletOptions): IServletProcess {
		debug('run servlet', entryPoint, options);
		const runServletProcess = () => {
			const servletProcess = this.baseServletRunner.run(entryPoint, options);
			if (options?.healthcheck?.enabled) {
				this.healthcheckServletProcess(servletProcess, options?.healthcheck);
			}
			return servletProcess;
		};
		const supervisedServletProcess = new SupervisedServletProcess(runServletProcess);
		this.servletProcesses.push(supervisedServletProcess);
		return supervisedServletProcess;
	}

	public async closeAll() {
		debug('close all servlets');
		await Promise.all(this.servletProcesses.map((servletProcess: SupervisedServletProcess) => servletProcess.stop()));
	}

	private healthcheckServletProcess(servletProcess: IServletProcess, healthcheckOptions?: IHealthcheckOptions) {
		debug('healthcheck servlet ' + servletProcess.getId());
		const timeoutMs = healthcheckOptions?.timeoutMs || this.defaultHealthcheckTimeoutMs;

		servletProcess.onMessage((message: IServletProcessMessage) => {
			if (message.type === this.messageTypePrefix + '_api.ready') {
				debug(`servlet ${servletProcess.getId()} ready, start healthchecking`);
				servletProcess.sendMessage({
					type: this.messageTypePrefix + '.config.healthcheck',
					healthcheck: {
						timeoutMs,
						...healthcheckOptions,
					},
				});
			}
		});

		let timeoutHandler: any;
		let stopped = false;

		function resetTimeoutRestart() {
			if (!stopped) {
				debug(`reset servlet ${servletProcess.getId()} healthcheck timeout`);
				clearTimeout(timeoutHandler);
				timeoutHandler = setTimeout(async () => {
					debug(`servlet ${servletProcess.getId()} healthcheck timed out and the servlet process will be stopped`);
					stopped = true;
					await servletProcess.stop();
				}, timeoutMs);
			}
		}

		servletProcess.onMessage((message: IServletProcessMessage) => {
			debug(`servlet ${servletProcess.getId()} message received`, message);
			if (message.type === this.messageTypePrefix + '.ping') {
				// type .ping used in messages of front-applet (later front-servlet) lib
				// see pingHandler
				resetTimeoutRestart();
			} else {
				// Not handled any other messages now
			}
		});

		servletProcess.onceClosed(() => {
			stopped = true;
			clearTimeout(timeoutHandler);
		});

		resetTimeoutRestart();
	}
}
