import { EventEmitter } from 'events';
import ISerial, { IOptions, ISerialPort, SerialPortEvent } from '../NativeDevice/Hardware/ISerial';
import { generateUniqueHash } from '@signageos/lib/dist/Hash/generator';
import { locked } from '../Lock/lockedDecorator';
import ISerialPortStore from './ISerialPortStore';

export default class SerialPortStore implements ISerialPortStore {
	private openSerialPorts: { [refid: string]: ISerialPort } = {};
	private eventEmitter: EventEmitter;

	constructor(private serial: ISerial) {
		this.eventEmitter = new EventEmitter();
	}

	@locked('SerialOpenStore')
	public async openSerialPort(options: IOptions): Promise<string> {
		const serialPort = await this.serial.openPort(options);
		const refid = this.generateUniqueRefid();
		this.openSerialPorts[refid] = serialPort;
		serialPort.on(SerialPortEvent.DATA, (data: Uint8Array) => {
			this.eventEmitter.emit('data', refid, data);
		});
		serialPort.on(SerialPortEvent.CLOSE, () => {
			this.eventEmitter.emit('close', refid);
		});
		return refid;
	}

	public async writeToSerialPort(refid: string, data: string | number[] | Uint8Array) {
		const serialPort = this.getSerialPortByRefidOrThrowError(refid);
		await serialPort.write(data);
	}

	@locked('SerialOpenStore')
	public async closeSerialPort(refid: string) {
		const serialPort = this.getSerialPortByRefidOrThrowError(refid);
		await serialPort.close();
		delete this.openSerialPorts[refid];
	}

	@locked('SerialOpenStore')
	public async closeAllSerialPorts() {
		await Promise.all(Object.keys(this.openSerialPorts).map((refid: string) => this.openSerialPorts[refid].close()));
		this.openSerialPorts = {};
	}

	public on(event: 'data', listener: (refid: string, data: Uint8Array) => void): void;
	public on(event: 'close', listener: (refid: string) => void): void;
	public on(event: 'data' | 'close', listener: (refid: string, data: Uint8Array) => void | ((refid: string) => void)): void {
		this.eventEmitter.addListener(event, listener);
	}

	public removeEventListener(event: 'data' | 'close', listener: (...args: any[]) => void) {
		this.eventEmitter.removeListener(event, listener);
	}

	private generateUniqueRefid() {
		let refid: string;
		do {
			refid = generateUniqueHash(10);
		} while (this.openSerialPorts[refid]);
		return refid;
	}

	private getSerialPortByRefidOrThrowError(refid: string) {
		const serialPort = this.openSerialPorts[refid];
		if (!serialPort) {
			throw new Error('refid not found');
		}
		return serialPort;
	}
}
