import { SerialPortStream } from '@serialport/stream';
import { autoDetect } from '@serialport/bindings-cpp';
// @ts-ignore
import { BindingInterface, OpenOptions } from '@serialport/bindings-interface';
import { debug } from '@signageos/lib/dist/Debug/debugDecorator';
import ISerial, { IOptions, ISerialPort, SerialPortEvent } from '../NativeDevice/Hardware/ISerial';
import { locked } from '../Lock/lockedDecorator';
import SerialPort from './SerialPort';
import { difference } from 'lodash';

const DEBUG_NAMESPACE = '@signageos/front-display:Serial:Serial';

export default class Serial implements ISerial {
	private openPorts: SerialPort[] = [];

	private readonly binding: BindingInterface;

	/**
	 * @param binding Defaults to platform bindings. Override for testing.
	 */
	constructor(binding: BindingInterface = autoDetect()) {
		this.binding = binding;
	}

	@locked('serial')
	@debug(DEBUG_NAMESPACE)
	public async openPort(options: IOptions): Promise<ISerialPort> {
		const hardwareSerialPort = await this.createAndOpenSerialPort(options);
		const serialPort = new SerialPort(hardwareSerialPort);
		this.openPorts.push(serialPort);
		serialPort.on(SerialPortEvent.CLOSE, () => {
			this.openPorts = this.openPorts.filter((openPort: SerialPort) => openPort !== serialPort);
		});
		return serialPort;
	}

	@locked('serial')
	@debug(DEBUG_NAMESPACE)
	public async closeAll(): Promise<void> {
		await Promise.all(this.openPorts.map((openPort: SerialPort) => openPort.close()));
		this.openPorts = [];
	}

	private createAndOpenSerialPort(options: IOptions) {
		const path = typeof options.device !== 'undefined' ? options.device : '/dev/ttyUSB0';
		const serialPortOptions = this.prepareSerialPortOptions(path, options);
		const serialPort = new SerialPortStream({
			binding: this.binding,
			...serialPortOptions,
			autoOpen: false,
		});
		return new Promise((resolve: (serialPort: SerialPortStream) => void, reject: (error: Error) => void) => {
			serialPort.open((error: Error | null) => {
				if (error) {
					reject(error);
				} else {
					resolve(serialPort);
				}
			});
		});
	}

	private prepareSerialPortOptions(path: string, options: IOptions) {
		// > The constructor no longer has an optional baudRate
		// > (The old default of 9600 was slow and usually not what's required.).
		// > Because of this we combined the path into the options object and made it required.
		// https://serialport.io/docs/guide-upgrade#serialport-package
		const serialPortOptions: OpenOptions = {
			path,
			baudRate: typeof options.baudRate !== 'undefined' ? options.baudRate : 9600,
		};
		if (typeof options.parity !== 'undefined') {
			serialPortOptions.parity = options.parity;
		}
		if (typeof options.databits !== 'undefined') {
			serialPortOptions.dataBits = options.databits as typeof serialPortOptions.dataBits;
		}
		if (typeof options.stopbits !== 'undefined') {
			serialPortOptions.stopBits = options.stopbits as typeof serialPortOptions.stopBits;
		}
		if (typeof options.rtscts !== 'undefined') {
			serialPortOptions.rtscts = options.rtscts;
		}
		return serialPortOptions;
	}
}

// Check the passed options and throw an error if some of them are not supported
export function validateSerialPortOptions(options: IOptions, supportedOptions: (keyof IOptions)[]) {
	// baudRate is required to open a serial port - throw an error if it is missing
	if (!options.baudRate) {
		throw new Error('The baudRate option is required');
	}
	// throw an error in case that some of the passed options are not supported on the particular platform
	const unsupportedOptions = difference(Object.keys(options), supportedOptions);
	if (unsupportedOptions.length > 0) {
		throw new Error(`The following options are not supported in the SerialPort implementation: ${unsupportedOptions.join(', ')}`);
	}
}
