import { call, fork } from 'redux-saga/effects';
import wait from '@signageos/lib/dist/Timer/wait';
import { SubscriptionType } from '../Display/IConfig';
import { withDependencies } from '../DI/dependencyInjection';

interface PeriodicTaskOptions {
	randomize?: boolean;
}

export function runPeriodicTaskSagaWhilePlatform(
	initDelayMs: number,
	periodMs: number,
	task: () => any,
	options: PeriodicTaskOptions = {},
) {
	return runPeriodicTaskSagaWhile(initDelayMs, periodMs, task, SubscriptionType.platform, options);
}

export function runPeriodicTaskSagaWhileOpen(initDelayMs: number, periodMs: number, task: () => any, options: PeriodicTaskOptions = {}) {
	return runPeriodicTaskSagaWhile(initDelayMs, periodMs, task, SubscriptionType.open, options);
}

function runPeriodicTaskSagaWhile(
	initDelayMs: number,
	periodMs: number,
	task: () => any,
	onlyWhenSubscriptionType: SubscriptionType,
	options: PeriodicTaskOptions = {},
) {
	let secondTelemetryDelayMs = periodMs;
	if (options.randomize) {
		const randomInitialMultiplier = 0.5 + Math.random(); // Random value +/- 50% of the initial delay
		initDelayMs = Math.round(initDelayMs * randomInitialMultiplier);
		secondTelemetryDelayMs = Math.round(Math.random() * periodMs);
	}

	return fork(
		withDependencies(['subscriptionType'], function* ({ subscriptionType: currentSubscriptionType }) {
			yield call(wait, initDelayMs);
			const doTheTask = onlyWhenSubscriptionType === currentSubscriptionType;

			function* invokeTask() {
				if (doTheTask) {
					try {
						yield task();
					} catch (error) {
						console.error(error);
					}
				}
			}

			// Run the initial telemetry after the initial delay
			yield invokeTask();

			// Do the second telemetry after the initial randomized period
			// to distribute the load on the server
			yield call(wait, secondTelemetryDelayMs);

			while (true) {
				yield invokeTask();
				yield call(wait, periodMs);
			}
		}),
	);
}

export function runPeriodicTaskSagaUntilSuccessWhilePlatform(initDelayMs: number, periodMs: number, task: () => any) {
	return runPeriodicTaskSagaUntilSuccessWhile(initDelayMs, periodMs, task, SubscriptionType.platform);
}

export function runPeriodicTaskSagaUntilSuccessWhileOpen(initDelayMs: number, periodMs: number, task: () => any) {
	return runPeriodicTaskSagaUntilSuccessWhile(initDelayMs, periodMs, task, SubscriptionType.open);
}

function runPeriodicTaskSagaUntilSuccessWhile(
	initDelayMs: number,
	periodMs: number,
	task: () => any,
	onlyWhenSubscriptionType: SubscriptionType,
) {
	return fork(
		withDependencies(['subscriptionType'], function* ({ subscriptionType: currentSubscriptionType }) {
			yield call(wait, initDelayMs);
			while (true) {
				const doTheTask = onlyWhenSubscriptionType === currentSubscriptionType;
				if (doTheTask) {
					try {
						yield task();
						break;
					} catch (error) {
						// Do nothing
					}
				}
				yield call(wait, periodMs);
			}
		}),
	);
}
