import { call, fork, put } from 'redux-saga/effects';
import { bindAndTakeEveryOnlyWhenPlatform } from '../Socket/socketActionCreator';
import { DeviceClient } from '@signageos/remote-desktop/es6/RemoteDesktop/DeviceClient/DeviceClient';
import { whenCapable } from '../Feature/capable';
import ManagementCapability from '../NativeDevice/Management/ManagementCapability';
import { withDependencies } from '../DI/dependencyInjection';
import {
	SetRemoteDesktopSettings,
	SetRemoteDesktopSettingsFailed,
	SetRemoteDesktopSettingsSucceeded,
} from '@signageos/actions/dist/Device/RemoteDesktop/remoteDesktopActions';
import { SagaIterator } from 'redux-saga';
import { ResolutionItem } from '../NativeDevice/ResolutionList';
import { whenResponsible } from '../Feature/responsible';
import Responsibility from '../Feature/Responsibility';
import { ManagementCapabilities } from '@signageos/common-types/dist/Device/Capabilities/ManagementCapabilities';

export const remoteDesktopSaga = whenResponsible(
	Responsibility.REMOTE_DESKTOP,
	whenCapable(ManagementCapabilities.REMOTE_DESKTOP, remoteDesktopSagaDefault),
);

function* remoteDesktopSagaDefault() {
	yield fork(
		whenCapable(
			ManagementCapability.SCREENSHOT_UPLOAD,
			withDependencies(['managementDriver'], function* ({ managementDriver }): SagaIterator {
				const duid: string = yield call([managementDriver, managementDriver.getDeviceUid]);
				const resolutionItem: ResolutionItem = yield call([managementDriver, managementDriver.getResolution]);
				let remoteDesktopClient: DeviceClient | null = null;
				yield bindAndTakeEveryOnlyWhenPlatform(SetRemoteDesktopSettings, function* (action: SetRemoteDesktopSettings) {
					try {
						if (action.enabled) {
							if (remoteDesktopClient) {
								yield remoteDesktopClient.stop();
							}
							remoteDesktopClient = new DeviceClient(
								async () => ({
									screenshotUri: (await managementDriver.screenshotUpload(action.serverUrl)).url,
									width: resolutionItem.width,
									height: resolutionItem.height,
								}),
								duid,
								managementDriver.input,
							);
							yield remoteDesktopClient.start({
								serverUrl: action.serverUrl,
								frequencyMs: action.frequencyMs,
							});
						} else {
							if (remoteDesktopClient) {
								yield remoteDesktopClient.stop();
								remoteDesktopClient = null;
							}
						}
						yield put<SetRemoteDesktopSettingsSucceeded>({
							type: SetRemoteDesktopSettingsSucceeded,
							uid: action.uid,
						});
					} catch (error) {
						yield put<SetRemoteDesktopSettingsFailed>({
							type: SetRemoteDesktopSettingsFailed,
							uid: action.uid,
						});
					}
				});
			}),
		),
	);
}
